{
    This file is part of RA3701Control

    RACAL RA3701 Control program

    Copyright (C) 2025, 2026 G. Perotti, I1EPJ, i1epj@aricasale.it

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/.

                                   * * *
    Main form unit
}

unit ura3701;

{$mode objfpc}{$H+}

{$INCLUDE defines.inc}

interface

uses
  Classes, SysUtils, FileUtil, RTTICtrls, SpinEx, Forms, Controls, Graphics,
  Dialogs, StdCtrls, Menus, ExtCtrls, Buttons, ComCtrls, serial, types, LCLType,
  ActnList, Spin, ComboEx, mouse, dateutils, math, userparms, UAbout, UGNU, UMAN,
  urxaddr, umanager, uenttestno, JButton, lNetComponents, FileInfo,
  DefaultTranslator, LNET, uparse, utcp, uscanwindow, strutils, uchscan,URXINFO,
  utestlist, umenu, ufrscan, XMLConf;

type

  { TRA3701 }

  TRA3701 = class(TForm)
    LBIFGSQ: TLabel;
    LBFR1: TLabel;
    LBFR2: TLabel;
    LBFR3: TLabel;
    LBRXNUM: TLabel;
    LBFR: TLabel;
    LBKHZT: TLabel;
    LBdb120: TLabel;
    LBdb20: TLabel;
    LBdb100: TLabel;
    LBdb80: TLabel;
    LBdb60: TLabel;
    LBdb40: TLabel;
    LBdbMW2: TLabel;
    LBdbMW3: TLabel;
    LBdbMW4: TLabel;
    LBdbMW5: TLabel;
    LBPOWER: TLabel;
    LBRAXXXX: TLabel;
    LSCALE10: TLabel;
    LSCALE11: TLabel;
    LSCALE12: TLabel;
    LSCALE13: TLabel;
    LBBITE: TLabel;
    LBIFG: TLabel;
    LSCALE4: TLabel;
    LSCALE3: TLabel;
    LSCALE5: TLabel;
    LSCALE6: TLabel;
    LSCALE7: TLabel;
    LSCALE8: TLabel;
    LSCALE9: TLabel;
    LBCHANUNITS: TLabel;
    B0: TJButton;
    B1: TJButton;
    BPLMI: TJButton;
    B2: TJButton;
    B3: TJButton;
    B4: TJButton;
    B5: TJButton;
    B6: TJButton;
    B7: TJButton;
    B8: TJButton;
    B9: TJButton;
    BAM: TJButton;
    BCHAN: TJButton;
    BSCAN: TJButton;
    BCW: TJButton;
    BENTER: TJButton;
    BFM: TJButton;
    BFREQ: TJButton;
    BISB: TJButton;
    BMETER: TJButton;
    BAGCP: TJButton;
    BBWM: TJButton;
    BBWP: TJButton;
    BMENU: TJButton;
    BRUNH: TJButton;
    BLS: TJButton;
    BAUX: TJButton;
    BFSK: TJButton;
    BSQL: TJButton;
    BMAN: TJButton;
    BAGCM: TJButton;
    BLSB: TJButton;
    BM1: TJButton;
    BM2: TJButton;
    BM3: TJButton;
    BM4: TJButton;
    BRCL: TJButton;
    BREM: TJButton;
    BADDR: TJButton;
    BSTORE: TJButton;
    BTUNEP: TJButton;
    BBFO: TJButton;
    BTUNEM: TJButton;
    BUSB: TJButton;
    CBXAGC: TComboBoxEx;
    CBXBW: TComboBoxEx;
    CBXMODE: TComboBoxEx;
    LBADDRTENS: TLabel;
    LBCHANTENS: TLabel;
    LBADDRUNITS: TLabel;
    IMRacalLogo: TImage;
    LSCALE1: TLabel;
    LSCALE2: TLabel;
    LBCHAN: TLabel;
    LBCOR: TLabel;
    LBCHANNEL: TLabel;
    LBdbMW1: TLabel;
    LBDBM: TLabel;
    LBPHN: TLabel;
    LBVOLUME: TLabel;
    Lbnetw: TLabel;
    LBAF: TLabel;
    LBTUNE: TLabel;
    LBBFO: TLabel;
    LBSLOW: TLabel;
    LBMEDIUM: TLabel;
    LBFAST: TLabel;
    LBLOCK: TLabel;
    LBVARIABLE: TLabel;
    LBRF: TLabel;
    LBMODE: TLabel;
    LBBW: TLabel;
    LBBFOF: TLabel;
    LBANT: TLabel;
    LBAGC: TLabel;
    LBDBuV: TLabel;
    LBdb0: TLabel;
    LBMUTE: TLabel;
    LBDISPLINE1: TLabel;
    LBDISPLINE2: TLabel;
    LBHFREC: TLabel;
    MCONTBITE: TMenuItem;
    MBSCANW: TMenuItem;
    MDETERRLIST: TMenuItem;
    MRSTATE: TMenuItem;
    Separator8: TMenuItem;
    MRXFREQSCAN: TMenuItem;
    MRESTSTATE: TMenuItem;
    MRESETALLSCF: TMenuItem;
    MSETALLSCF: TMenuItem;
    MFFBITE: TMenuItem;
    Separator3: TMenuItem;
    MRFAMPOFF: TMenuItem;
    MRFAMPON: TMenuItem;
    MRFAMPAUTO: TMenuItem;
    Separator5: TMenuItem;
    MRFAMP: TMenuItem;
    MSHOWHAMLIB: TMenuItem;
    Separator7: TMenuItem;
    MUSEPROGCH: TMenuItem;
    MSAVEAUX: TMenuItem;
    Separator6: TMenuItem;
    MUPFCHANLIST: TMenuItem;
    Separator4: TMenuItem;
    MRXINFO: TMenuItem;
    MLOCKRX: TMenuItem;
    MSG: TMemo;
    PCENTER: TPanel;
    PRIGHT: TPanel;
    Separator2: TMenuItem;
    MTCP: TMenuItem;
    RA3701Server: TLTCPComponent;
    MMAGALLF: TMenuItem;
    Separator1: TMenuItem;
    MPORTPROL: TMenuItem;
    MMAG100: TMenuItem;
    MMAG110: TMenuItem;
    MMAG125: TMenuItem;
    MMAG150: TMenuItem;
    MMAG175: TMenuItem;
    MMAG200: TMenuItem;
    N2: TMenuItem;
    MFONTMAGN: TMenuItem;
    N1: TMenuItem;
    MTIMED: TMenuItem;
    M0P5KHZT: TMenuItem;
    MBITEUCONF: TMenuItem;
    MFACTBITE: TMenuItem;
    MOTHER: TMenuItem;
    RXFreq1H: TLabel;
    RXFreq1M: TLabel;
    RXFreq10H: TLabel;
    RXFreq100H: TLabel;
    RXFreq10M: TLabel;
    RXFreqDOT: TLabel;
    RXFreq1K: TLabel;
    RXFreq10K: TLabel;
    RXFreq100K: TLabel;
    LBMASTER: TLabel;
    LBADDR: TLabel;
    LBFREQ: TLabel;
    LBKHZ: TLabel;
    Shape1: TShape;
    Shape2: TShape;
    SHMEMOBAK: TShape;
    SHIFG: TShape;
    SHB12: TShape;
    SHB11: TShape;
    SHB10: TShape;
    SHB9: TShape;
    SHB8: TShape;
    SHB7: TShape;
    SHB1: TShape;
    SHB5: TShape;
    SHB6: TShape;
    SHB2: TShape;
    SHB3: TShape;
    Shape7: TShape;
    SHB4: TShape;
    Shape9: TShape;
    SHVOL1: TShape;
    SHVOL: TShape;
    SHRFG: TShape;
    SHSWB1: TShape;
    SHSWB2: TShape;
    SHSW: TShape;
    SHPH: TShape;
    SPFREQ: TSpinEditEx;
    LBWARN: TLabel;
    LBREMOTE: TLabel;
    LBSCAN: TLabel;
    M1KHZT: TMenuItem;
    M10KHZT: TMenuItem;
    M100KHZT: TMenuItem;
    MSETKHZT: TMenuItem;
    PLEFT: TPanel;
    MenuItem3: TMenuItem;
    MMANCH: TMenuItem;
    MMANST: TMenuItem;
    MSCHANN: TMenuItem;
    MRXADD: TMenuItem;
    MenuItem11: TMenuItem;
    MChannels: TMenuItem;
    MSHOWC: TMenuItem;
    MBlack: TMenuItem;
    MBlue: TMenuItem;
    MYellow: TMenuItem;
    MenuItem6: TMenuItem;
    MenuItem7: TMenuItem;
    MLEDC: TMenuItem;
    MenuItem9: TMenuItem;
    MENALL: TMenuItem;
    MGNULIC: TMenuItem;
    MABOUT: TMenuItem;
    MMAN: TMenuItem;
    MHelp: TMenuItem;
    MLSTATE: TMenuItem;
    MSSTATE: TMenuItem;
    MenuItem5: TMenuItem;
    MSmeter: TMenuItem;
    OpenDialog1: TOpenDialog;
    SaveDialog1: TSaveDialog;
    SHKNOB1: TShape;
    SHKNOB: TShape;
    LBFAULT: TLabel;
    LBRAXXXX1: TLabel;
    SPRXNUM: TSpinEditEx;
    TBITE: TTimer;
    TUPDDISP: TTimer;
    TNET: TTimer;
    TSmeter: TTimer;
    TSMDISP: TTimer;
    MainMenu1: TMainMenu;
    MFile: TMenuItem;
    MExit: TMenuItem;
    MOptions: TMenuItem;
    MSERPARMS: TMenuItem;
    XMLConfig1: TXMLConfig;
    procedure BADDRClick(Sender: TObject);
    procedure BAGCMClick(Sender: TObject);
    procedure BAGCPClick(Sender: TObject);
    procedure BAUXClick(Sender: TObject);
    procedure BBWMClick(Sender: TObject);
    procedure BBWPClick(Sender: TObject);
    procedure BCHANClick(Sender: TObject);
    procedure BFSKClick(Sender: TObject);
    procedure BLSClick(Sender: TObject);
    procedure BM1Click(Sender: TObject);
    procedure BM2Click(Sender: TObject);
    procedure BM3Click(Sender: TObject);
    procedure BM4Click(Sender: TObject);
    procedure BMANClick(Sender: TObject);
    procedure BMENUClick(Sender: TObject);
    procedure BMETERClick(Sender: TObject);
    procedure BRUNHClick(Sender: TObject);
    procedure BSCANClick(Sender: TObject);
    procedure BCWClick(Sender: TObject);
    procedure BSQLClick(Sender: TObject);
    procedure BTUNEMClick(Sender: TObject);
    procedure BTUNEPClick(Sender: TObject);
    procedure BENTERClick(Sender: TObject);
    procedure BSTOREClick(Sender: TObject);
    procedure BAMClick(Sender: TObject);
    procedure B4Click(Sender: TObject);
    procedure B7Click(Sender: TObject);
    procedure B8Click(Sender: TObject);
    procedure B9Click(Sender: TObject);
    procedure BFMClick(Sender: TObject);
    procedure BLSBClick(Sender: TObject);
    procedure B5Click(Sender: TObject);
    procedure B6Click(Sender: TObject);
    procedure B2Click(Sender: TObject);
    procedure B3Click(Sender: TObject);
    procedure B0Click(Sender: TObject);
    procedure BREMClick(Sender: TObject);
    procedure BRCLClick(Sender: TObject);
    procedure BBFOClick(Sender: TObject);
    procedure BFREQClick(Sender: TObject);
    procedure BISBClick(Sender: TObject);
    procedure BUSBClick(Sender: TObject);
    procedure CBXAGCChange(Sender: TObject);
    procedure CBXBWChange(Sender: TObject);
    procedure CBXMODEChange(Sender: TObject);
    procedure CHANMouseUp(Sender: TObject; Button: TMouseButton;
        Shift: TShiftState; X, Y: Integer);
    procedure CHANMouseWheel(Sender: TObject; Shift: TShiftState;
        WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
    procedure FormClose(Sender: TObject; var CloseAction: TCloseAction);
    procedure FormKeyPress(Sender: TObject; var Key: char);
    procedure FormResize(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure MBSCANWClick(Sender: TObject);
    procedure MCONTBITEClick(Sender: TObject);
    procedure MDETERRLISTClick(Sender: TObject);
    procedure MFFBITEClick(Sender: TObject);
    procedure MLOCKRXClick(Sender: TObject);
    procedure MSERPARMSClick(Sender: TObject);
    procedure MRESETALLSCFClick(Sender: TObject);
    procedure MRXFREQSCANClick(Sender: TObject);
    procedure MRXINFOClick(Sender: TObject);
    procedure MSAVEAUXClick(Sender: TObject);
    procedure MSETALLSCFClick(Sender: TObject);
    procedure MTCPClick(Sender: TObject);
    procedure MUPFCHANLISTClick(Sender: TObject);
    procedure RA3701ServerDisconnect(aSocket: TLSocket);
    procedure SHPHMouseEnter(Sender: TObject);
    procedure SHRFGMouseEnter(Sender: TObject);
    procedure SHRFGMouseWheel(Sender: TObject; Shift: TShiftState;
      WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
    procedure SHSWMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure SHVOLMouseEnter(Sender: TObject);
    procedure SHVOLMouseWheel(Sender: TObject; Shift: TShiftState;
      WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
    procedure SPRXNUMChange(Sender: TObject);
    procedure TBITETimer(Sender: TObject);
    procedure RA3701ServerAccept(aSocket: TLSocket);
    procedure RA3701ServerError(const amsg: string; aSocket: TLSocket);
    procedure RA3701ServerReceive(aSocket: TLSocket);
    procedure MABOUTClick(Sender: TObject);
    procedure MENALLClick(Sender: TObject);
    procedure MKHZTClick(Sender: TObject);
    procedure MMAGALLFClick(Sender: TObject);
    procedure MMANCHClick(Sender: TObject);
    procedure MMANSTClick(Sender: TObject);
    procedure MRSTATEClick(Sender: TObject);
    procedure MSGDblClick(Sender: TObject);
    procedure MBlueClick(Sender: TObject);
    procedure MBlackClick(Sender: TObject);
    procedure MGNULICClick(Sender: TObject);
    procedure MLSTATEClick(Sender: TObject);
    procedure MMANClick(Sender: TObject);
    procedure MExitClick(Sender: TObject);
    procedure MSHOWCClick(Sender: TObject);
    procedure MSSTATEClick(Sender: TObject);
    procedure MSmeterClick(Sender: TObject);
    procedure MYellowClick(Sender: TObject);
    procedure MBITEUCONFClick(Sender: TObject);
    procedure MFACTBITEClick(Sender: TObject);
    procedure RXfreqMouseUp(Sender: TObject; Button: TMouseButton;
            Shift: TShiftState; X, Y: Integer);
    procedure RXFreqMouseWheel(Sender: TObject; Shift: TShiftState;
        WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
    procedure SHKNOB1MouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure SHKNOBMouseUp(Sender: TObject; Button: TMouseButton;
        Shift: TShiftState; X, Y: Integer);
    procedure SPFREQChange(Sender: TObject);
    procedure SPFREQMouseWheel(Sender: TObject; Shift: TShiftState;
        WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
    procedure TSMDISPTimer(Sender: TObject);
    procedure TSmeterTimer(Sender: TObject);
    procedure B1Click(Sender: TObject);
    procedure FormMouseWheel(Sender: TObject; Shift: TShiftState;
            WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
    procedure FormCreate(Sender: TObject);
    procedure KeyboardHandler(keyp: string);
    procedure DispRXf;
    function SendCommand(cmd, par: string): boolean;
    function OpenRemote: boolean;
    function CloseRemote: boolean;
    function SetRXFreq: boolean;
    procedure GetStatus;
    procedure LoadChannel(Sender: TObject);
    procedure RestoreState;
    procedure SaveConfig;
    procedure ReadConfig;
    procedure DisableAllControls;
    procedure EnableAllControls;
    procedure SetActiveButtons;
    procedure EnableAllButtons;
    procedure CloseProgram;
    procedure LoadChannelsDir;
    procedure UpdateDisplay;
    function ParseStatus: boolean;
    procedure DisableAllButtons;
    procedure DisableKeypad;
    procedure EnableKeypad;
    procedure EnableCmdKeys;
    procedure EnableCombosAndEdits;
    procedure SpinEditUpdateOnly(Sender: TObject; value: integer);
    function AdjustDecimalSeparator(s: string): string;
    function GetReply: boolean;
    function GetAck: boolean;
    procedure ReadChanFile;
    procedure SaveChanFile;
    procedure SetColors(AColor: TCOlor);
    function GetLevel: integer;
    function GetRxstring: string;
    procedure PutRXString(s: string);
    function GetChanString: string;
    procedure PutChanString(s: string);
    procedure PutAddrString(s: string);
    procedure MFONTMAGNClick(Sender: TObject);
    procedure FontMagnify;
    function ShowControlChars(cmd: string): string;
    procedure TUPDDISPTimer(Sender: TObject);
    procedure UpdateLeftPanel;
    procedure UpdateCenterPanel;
    procedure UpdateRightPanel;
    procedure FormatDisplayLine1(mode,agcm,agct,sq:string);
    procedure FormatDisplayLine2(bw,bfo,ant:string);
    procedure UpdateIFGainAndSquelch;
    procedure UpdateSpinAndCombos;
    procedure LoadLastState;
    procedure SaveLastState;
    procedure DoBite(BITELevel: integer);
    procedure SetMainFormCaption;
    procedure SetDefaultConfig;
    procedure DoMessages;
    procedure SetBWCombo;
    procedure UpdateSquelchLabel;
    procedure ReleaseKBLock;
    procedure MRFAMPXClick(Sender: TObject);
    function InterpolateRFLevel(RFLevel: integer): real;
    function CalcCRC(ms: string): string;
    function GetAddrString: string;
  private
    { private declarations }
    procedure Message(ms: string);
  public
    { public declarations }
    property MsgToShow: string write Message;
  end;

Const
  //
  // Program name amd copyright
  //
  PROGNAME: string = 'RA3701Control';
  COPYRIGHTST = ' (C) 2025, 2026 G. Perotti, I1EPJ';
  //
  // RA3701 remote commands
  //
  Get_All:         string = 'QALL';  // Request all settings

  Get_AFLevel:     string = 'QAFL';   // returns AFLxxx xxx=000..255
  Set_AGC:         string = 'AGC';    // + <agc>,<time> e.g. AGC0,1
  Get_AGC:         string = 'QAGC';   //
  Set_Ant:         string = 'ANT';    // + <ant num>, e.g. ANT03
  Get_Ant:         string = 'QANT';   //
  Set_Audio:       string = 'AUDIO';  // + <state>,0 e.g. AUDIO1,0
  Get_Audio:       string = 'QAUDIO'; //
  Set_BFO:         string = 'BFO';    // + <bfo freq> e.g. BFO-800
  Get_BFO:         string = 'QBFO';   //
  Set_IFBW:        string = 'B';      //+ <type>,<bwno>, e.g. B1,10 See also QBCON.
  Get_IFBW:        string = 'QB';     //
  Set_BWConf:      string = 'BCON';   // + bandwidth data, see manual pag. 5-21
  Get_BWConf:      string = 'QBCON';  //
  Set_BITE:        string = 'BITE';   // + <level>,<test num>,<message>
  Get_BITE:        string = 'QBITE';  //
  Get_BWList:      string = 'QBWL';   // + returns BW list see manual pag 5-23
  Set_COR:         string = 'C';      // + <level>,<inhibit> e.g. cC0,0
  Get_COR:         string = 'QC';     //
  Set_Chan:        string = 'CHAN';   // + <chan no> e.g. CHAN01
  Get_Chan:        string = 'QCHAN';  //

  Clear_AllChans:  string = 'CLEARALLCHANNES'; // no params
  Change_ScanFlgs: string = 'CHGF';   // + flag value e.g. CHGF1 (changes ALL channels)

  Set_CORSQLevel:  string = 'CORL';   // + <COR level 000..255> e.g. CORL128
  Get_CORSQLevel:  string = 'QCORL';  //
  Set_CORControl:  string = 'COR';    // + <enable>,<hang> e.g. COR1,42
  Get_CORControl:  string = 'QCOR';   //
  Set_DivMaster:   string = 'DIVM';   // + <0|1> e.g. DIVM0
  Get_DivMaster:   string = 'QDIVM';  //
  Set_DeltaMode:   string = 'DLTA';   // no parameters
  Enter:           string = 'ENTR';   // same as ENTER key
  Set_Frequency:   string = 'F';      // + frequency e.g. F13264000
  Get_Frequency:   string = 'QF';     //
  Get_Fault:       string = 'QFAULT'; // Get fault status, reply FAULT0 if none
  Set_ScanFlag:    string = 'FLG';    // + 0=reset, 1=set e.g. FLG1
  Get_ScanFlag:    string = 'QFLG';   //
  Set_IFGain:      string = 'G';      // + <gain 000..255> e.g. G128
  Get_IFGain:      string = 'QG';     //
  Get_EQID:        string = 'QID';    // returns receiver id see manual pag. 5-26
  Set_LinkParms:   string = 'LINK';   // see manual pag. 5-26..5-28
  Get_LinkParms:   string = 'QLINK';  // [+ <port>], default tributary
  Set_Mode:        string = 'M';      // + <mode> e.g. M5
  Get_Mode:        string = 'QM';     //
  Set_Mute:        string = 'MUTE';   // + 0=no 1=yes e.g. MUTE1
  Get_Mute:        string = 'QMUTE';  //
  Set_PBT:         string = 'PASSB';  // + <type>,<bwno> ?
  Get_PBT:         string = 'QPASSB'; //
  Set_PBTFreq:     string = 'PASSF';  // + <PBT freq>, e.g. PASSF-800 (as BFO)
  Get_PBTFreq:     string = 'QPASSF'; //
  Recall:          string = 'RCL';    // same as RCL key
  Set_Remote:      string = 'REM';    // + <0=local|1 or 2=remote> e.g. REM1
  Set_RFAmp:       string = 'RFAMP';  // +<0=off|1=on|2=auto> e.g. RFAMP0
  Get_RFAmp:       string = 'QRFAMP'; //
  Get_RFLevel:     string = 'QRFL';   // returns RFLnnn nnn = 000,,255
  SubAddr_Pref:    string = 'S';      // for diversity control
  Set_Scan:        string = 'SCAN';   // + <type><dir> e.g. SCAN10
  Get_Scan:        string = 'QSCAN';  //
  Set_FrSwParms:   string = 'SCFR';   // + 4 params, see manual pag. 5-31
  Get_FrSwParms:   string = 'QSCFR';  // See page 5-31
  Set_ScanParms:   string = 'SCCH';   // + 4 params, see manual pag. 5-31
  Get_ScanParms:   string = 'QSCCH';  //
  Set_SerialNo:    string = 'SN';     // + "serial number"
  Get_SerialNo:    string = 'QSN';    //
  Set_Squelch:     string = 'SQU';    // + <0=off|1=on>
  Get_Squelch:     string = 'QSQU';   //
  Store_Channel:   string = 'STRE';   // + <chan no 00..99>
  Set_SWProgNo:    string = 'SW';     // + "number","issue"
  Get_SWProgNo:    string = 'QSW';    //
  Enter_FreeTune:  string = 'T';      // + <tune mode>

  // Revertive commands preamble
  RevCmds: array of string = (
    'AFL',
    'AGC',
    'ANT',
    'AUDIO',
    'BFO',
    'BCON',
    'BITE',
    'BWL',
    'B',      // Must be after all other Bxxx commands to avoid false match
    'CHAN',
    'CORL',
    'COR',
    'C',      // As above
    'DIVM',
    'DLTA',
    'FLG',
    'FAULT',
    'F',      // As above
    'G',
    'ID',
    'LINK',
    'MUTE',
    'M',      // As above
    'PASSB',
    'PASSF',
    'REM',
    'RFAMP',
    'RFL',
    'SCAN',
    'SCCH',
    'SN',
    'SQU',
    'SW',
    'ERR'     // Error message
);

  // Filenames
  XMLConfigFileName: string = 'Config.xml';
  RXChanFileName: string = 'RXChan.dat';
  AUXFilename:    string = 'AUXconf.dat';
  StateFileName:  string = 'Laststate.dat';

  // Mode values
  M_AUX  = 0;  // manual says so...
  M_USB  = 1;
  M_LSB  = 2;
  M_AM   = 3;
  M_FM   = 4;
  M_CW   = 5;
  M_FSK  = 6;  // ditto
  M_ISBU = 7;  // ditto
  M_ISBL = 8;  // ditto

  // Bandwidth types
  BWTYPE_USB = 1;
  BWTYPE_LSB = 2;
  BWTYPE_SYM = 3;

  // AGC modes
  AGCON = 0;
  MANONLY  = 1;
  MANT = 2;

  // AGC Time Constant
  AGC_SHORT = 0;
  AGC_MEDIUM = 1;
  AGC_LONG = 2;
  AGC_L2DATA = 3;
  AGC_L2NORMAL = 4;

  // RFAMP States
  RFAMP_OFF = 0;
  RFAMP_ON = 1;
  RFAMP_AUTO = 2;

  // frequency coverage
  MINRXF = 0;
  MAXRXF = 29999999;

  // Remote Modes
  LOCAL = '0';
  REMOTE_NOKBLOCK = '1';
  REMOTE_KBLOCK = '2';

  // LCC bit names and masks
  OUTPUT_READY = $01;
  OUTPUT_PHASE = $02;
  INPUT_ACCEPT = $04;
  INPUT_PERMIT = $08;
  INPUT_PHASE  = $10;
  ALWAYS_ZERO  = $20;
  ALWAYS_ONE   = $40;
  PARITY_IFTX  = $80;


  // Special characters
  CR: char = #$0d;
  LF: char = #$0a;
  XON: char = #$11;
  XOFF: char = #$13;
  stCR: string = #$0d;
  stLF: string = #$0a;

  // TRUE/FALSE Constants
  ENABLED:  boolean = TRUE;
  DISABLED: boolean = FALSE;
  NOERROR:  boolean = TRUE;
  ERROR:    boolean = FALSE;
  NUMKEYS:  boolean = TRUE;
  FUNKEYS:  boolean = FALSE;
  RFMETER:  boolean = FALSE;
  AFMETER:  boolean = TRUE;

  // Timeout & delay values
  READTIMEOUT = 3000 * OneMillisecond;
  OPENTIMEOUT = 3000 * OneMillisecond;
  CTSTIMEOUT = 500 * OneMillisecond;
  DELAY100MS = 100 * OneMillisecond;
  REPLYTIMEOUT = 300 * OneMillisecond;

  // Keyboard states
  STATE_POWEROFF = 0;
  STATE_NORMAL   = 1;
  STATE_SETFREQ  = 2;
  STATE_SETBFO   = 3;
  STATE_SELCHAN  = 4;
  STATE_STOCHAN  = 5;
  STATE_SCAN     = 6;
  STATE_BITECONF = 7;
  STATE_BITEFF   = 8;
  STATE_BITEFACT = 9;
  STATE_BITESTOP = 10;
  STATE_BITEEND  = 11;
  STATE_SHOWADDR = 12;
  STATE_SETADDR  = 13;
  STATE_MENU     = 90;
  STATE_MENU_M1  = 91;
  STATE_MENU_M2  = 92;
  STATE_MENU_M3  = 93;
  STATE_MENU_M4  = 94;

  // Revertive command codes
  AFL   = 0;
  AGC   = 1;
  ANT   = 2;
  AUDIO = 3;
  BFO   = 4;
  BCON  = 5;
  BITE  = 6;
  BWL   = 7;
  B     = 8;
  CHAN  = 9;
  CORL  = 10;
  COR   = 11;
  C     = 12;
  DIVM  = 13;
  DLTA  = 14;
  FLG   = 15;
  FAULT = 16;
  F     = 17;
  G     = 18;
  ID    = 19;
  LINK  = 20;
  MUTE  = 21;
  M     = 22;
  PASSB = 23;
  PASSF = 24;
  REM   = 25;
  RFAMP = 26;
  RFL   = 27;
  SCAN  = 28;
  SCCH  = 29;
  SN    = 30;
  SQU   = 31;
  SW    = 32;
  ERR   = 33;

  // Display background color
  DispBackColor: TColor = $085dcc;

  // Display foreground color
  DispForeColor:TColor = $0000FDFF;

  // Object resize array max components number
  MAXCOMPONENTS = 300;

  // Line separation
  LINESEP = '—————————————————————————————————————————————————————';

  // PAD for commands
  pad = '                                 ';

  // Control characters unicode symbols array for ShowControlChars()
  ControlImages: array[0..31] of string = (
  '␀','␁','␂','␃','␄','␅','␆','␇','␈','␉','␊','␋','␌','␍','␎','␏',
  '␐','␑','␒','␓','␔','␕','␖','␗','␘','␙','␚','␛','␜','␝','␞','␟' );


Type
  // The RX state
  TRXState = record
    AFLevel: integer;
    AGCMode: integer;
    AGCTimeCon: integer;
    Ant: integer;
    Audio: boolean;
    Bandwidth: integer;
    BFO: integer;
    BITEFaults: integer;
    BITELevel: integer;
    BITETest: integer;
    BWNumber: integer;
    BWType: integer;
    COREnabled: boolean;
    CORHangtime: integer;
    CORLevel: boolean;
    CORSQLevel: integer;
    DivMaster: boolean;
    Freq_rx: longint;
    Freq_step: integer;
    IFGain: integer;
    MeterMode: boolean;
    Mode_RX: integer;
    Mute: boolean;
    PASSBType: integer;
    PASSBNumber: integer;
    PASSBFreq: integer;
    RFAmp: integer;
    RFLevel: integer;
    Scan: boolean;
    ScanFlag: boolean;
    ScanInhibit: boolean;
    SquelchEnabled: boolean;
    State: integer;
    Tune: boolean;
  end;

  // RX channels (program version)
  TRXChannels = array[0..99] of TRXState;

  // The bandwidths the receiver is capable of
  TBandwidth = record
    BWType: integer;
    BWNumber: integer;
    Position: integer;
    FLow: integer;
    FHigh: integer;
    Bandwidth: integer;
    FOffset: integer;
  end;

  // lcc bits
  TLCC = record
    Output_Ready: boolean;
    Output_phase: boolean;
    Input_Accept: boolean;
    Input_Permit: boolean;
    Input_Phase: boolean;
    Parity: boolean;
  end;

var
  RA3701: TRA3701;
  ProgVersion: TVersionQuad;
  Major, Minor, Revision, Build: integer;
  ShortVersion, LongVersion: string;
  RXChannels: TRXChannels;
  RXState, OldRXState: TRXState;
  SerPort: TSerialHandle = 0;
  SleepTime: integer;
  TimeOneCharMs: integer;
  CurRXAddress: integer;
  s,t, dBm: integer;
  Status: string = '';
  sCHtmp: string ='  ';
  iChnum: integer;
  LastChan: integer = 1;
  rCurChan: real;
  CurChan: integer;
  Enable_status: boolean = FALSE;
  In_Wheel: boolean = FALSE;
  In_RXFreqMouseUp: boolean = FALSE;
  In_TXFreqMouseUp: boolean = FALSE;
  In_ChanMouseWheel: boolean = FALSE;
  In_ChanMouseUp: boolean = FALSE;
  In_KnobMouseMove: boolean = FALSE;
  In_Command: boolean = FALSE;
  In_ChanUpdate: boolean = FALSE;
  Update_Only: boolean = FALSE;
  tmp: longint;
  HomeDir,StateDir,BandsDir,ChannelsDir,BandScansDir,DocDir, GPLv3Dir: string;
  timeout: TTime;
  LEDColorOFF, LEDColorON: TColor;
  sLEDColor: string;
  BandFileName: string;
  RemoteMode: string;
  CmdTag: integer = -1;
  PrevTag: integer = -1;
  DigitCoord: array[1..9, 1..2] of integer;
  sant,sf1,sd,si,sm,sb,sa,sx,sr,ss: string;
  KeyboardState: integer;
  DigitWidth: integer;
  prevalpha: real = 0;
  r, cx, cy,X2, Y2, R2: real;
  KnobOffsetX, KnobOffsetY: integer;
  HzPerTurn: integer = 1000;
  OKPressed: boolean;
  ScanDelay: TTime = 2*Delay100ms;
  LastTuningTime: TTime;
  PPI_X, PPI_Y: integer;
  FontMagn: integer = 100;
  ErrMsg: string;
  Remote_Enabled: boolean;
  FaultDetected: boolean = FALSE;
  OldTSEn,OldTBEn: boolean;
  RXId, RXModel,FWVersion,RXSN: string;
  Bandwidths: array[0..47] of TBandwidth;  // 48 bw, 16 for USB, 16 for LSB and 16 for SYM filters
  USBBandwidths: array[0..15] of TBandwidth;
  LSBBandwidths: array[0..15] of TBandwidth;
  SYMBandwidths: array[0..15] of TBandwidth;
  NumBandwidths, BWNum, NumUSBBW, NumLSBBW, NUMSYMBW: integer;
  LinkParms: string;
  MenuLevel: integer;
  snexttest, slevel: string;
  OldCurChan: integer;
  StartSc: boolean = FALSE;
  CurRXNum: integer;
  LCCBits: TLCC;

  {$IFDEF AUTOSCALE}
  // Variables used by autoscale code
  OH, OW: integer;
  Xs: array[0..MAXCOMPONENTS] of integer;
  Ys: array[0..MAXCOMPONENTS] of integer;
  Ws: array[0..MAXCOMPONENTS] of integer;
  Hs: array[0..MAXCOMPONENTS] of integer;
  Wt: array[0..MAXCOMPONENTS] of Integer;
  Ht: Integer;
  Wc: array[0..MAXCOMPONENTS] of Integer;
  Hc: Integer;
  Wn: array[0..MAXCOMPONENTS] of Integer;
  Hn: Integer;
  Hf: array[0..MAXCOMPONENTS] of Integer;
  {$ENDIF}
  PrevHeight, PrevWidth: integer;
  StartHeight, StartWidth: integer;
  StartX, StartY: integer;

  // SMeter shapes array
  SMeter: array[1..12] of TShape;

  // Variables for TNET rigctl server
  RXLine: string;
  NumTCPConnections: integer = 0;

  // Other miscellaneous flags
  RemFreqChange: boolean = FALSE;
  PanelUpdateRequired: boolean = FALSE;
  DeltaMode_ON: boolean = FALSE;
  RCLPressed: boolean = FALSE;
  FaultFindingBITE: boolean = FALSE;
  MsgToShow: string  = '';

resourcestring
// Declared here for i18n
         ADDRERR = 'Invalid address %d, expected %d';
           APPLY = 'Apply';
   AUXFILELOADED = 'AUX mode file loaded.';
  BITECOMPLFAULT = 'BITE level %s completed, failed: last failed test #%s (%s,%s).';
BITECOMPLNOFAULT = 'BITE level %s completed, no faults.';
   BITELABFORMAT = '%s test #%s (%s, %s): %s';
    BITELEVDESCR = 'level %4d';
          BWCONF = 'Bandwidths configuration:';
    CANTOPENCHAN = 'Can''t open channel file.';
    CANTOPENFILE = 'Can''t open file.';
    CANTREADCHAN = 'Can''t read channels file.';
CANTWRITEAUXFILE = 'Can''t write AUX file.';
   CANTREADSTATE = 'Can''t read last state file.';
   CANTWRITECHAN = 'Can''t write channels file.';
 CANTWRITECONFIG = 'Can''t write config file.';
  CANTWRITESTATE = 'Can''t write state file.';
         CHANMGR = 'Channel Manager';
       CHECKINST = 'Check your installation.';
          CLOSEW = 'Close';
        CLOSEPGM = 'Close program';
             CMD = 'Cmd: ';
   COMMANDFAILED = 'Command failed.';
 COMMANDNOTFOUND = 'Command not found: ';
    COMMANDNOTEX = 'Command ''%s'' not executed: %s';
        CONFBITE = 'Confidence BITE';
  CONFIGNOTFOUND = 'Config file not found, loading and saving'+LineEnding+
                   'default configuration.';
      CORCONTROL = 'COR';
        CORLEVEL = 'COR level';
          CRCERR = 'CRC error: got %s, expected %s';
     DEFAULTSTEP = 'Default step ';
          DIRECT = 'Direction';
    DOCFILESNOTF = 'Manual and/or license files not found.';
            DONE = 'Done.';
   DONTWORKINMAN = 'NOTE: Squelch does not work in MAN modes.';
       DWELLTIME = 'Dwell time';
      ENDERRLIST = '--- End error list ---';
     ENTERCHANNO = 'Enter channel# (0..99) or use tuning knob or mouse wheel to select.';
  ENTERSCANPARMS = 'Enter scan parameters';
    ENTERFREQKHZ = 'Enter frequency in kHz.';
        ERRLEVEL = 'Error reading levels, reading disabled.';
     ENTERRXADDR = 'Enter RX address, ENTER when done, RCL to exit';
    ERRORLISTMSG = 'Test#: %s %s';
        ERRORMSG = 'Error';
        ERRORNET = 'Net error occurred: %s';
ERRORREADINGCONF = 'Error reading config file';
        EXECBITE = 'BITE Level %1D, test #%s (%s, %s). Result: %s';
 EXECUTINGTESTNO = 'Executing test #%s (%s, %s)';
        FACTBITE = 'Factory BITE';
          FFBITE = 'Fault finding BITE';
          FAILED = ' failed.';
 FAILEDINITCOMMS = 'Failed to initialize communications.';
       FIRMWVERS = 'FW Version     : ';
       FREQSTART = 'Start frequency';
        FREQSTOP = 'Stop frequency';
        FREQSTEP = 'Frequency step';
    GETPROGVFAIL = 'GetProgramVersion failed, please enable'+LineEnding
                  +'version informations in Project Options.';
        GOTFAULT = 'Got FAULT revertive command.';
    GOTPWRONBITE = 'Got power-on BITE message: ';
   INTINITFAILED = 'Interface init failed.';
            KHZT = 'kHz/t';
      LINKPARAMS = 'Link params    : ';
    LISTENFAILED = 'RA3701server socket: listen() failed.';
  LOADINGLASTSTA = 'Loading last state file.';
      LOADINGSTA = 'Loading state...';
       LOADINGCH = 'Loading channel...';
   MENUNOTIMPLL1 = '    Menu system     '; // 20 chars
   MENUNOTIMPLL2 = 'not yet implemented '; // 20 chars
      MRXADDCAPT = 'Receiver address (%d)';
           NOACK = 'Command not acknowledged, reason:';
       NOAUXFILE = 'No AUX file saved.';
       NOCHANDEF = 'No channels defined';
      NOCHANFILE = 'No channels file found; consider updating channels list.';
     NOCHANSCANF = 'No channels with SCAN flag set found.';
        NOFAULTS = 'No faults.';
    NOLEVELREPLY = 'No reply to QRFL command, level reading disabled.';
      NOMANIFGSQ = 'NOTE: MAN AGC or squelch not active.';
    NOMOREFAULTS = 'No more faults.';
        NOREMVOL = 'No remote commands to set volume.';
         NOREMRH = 'No remote command for RUN/HOLD key.';
         NOREPLY = 'Reply not received.';
         NOTIMPL = 'Not yet implemented.';
      NOVERSINFO = 'No version info.';
   PARSESTFAILED = 'ParseStatus failed.';
          PASSED = 'passed.';
  PRESSRCLTOTERM = 'Press RCL to terminate BITE.';
  PRESSRCLTOCONT = 'Press RCL to continue.';
  PRESSRCLTOSTOP = 'Press RCL to stop BITE.';
      READBWLIST = 'BW list read: found %d bandwidths.';
      READCHANNO = 'Read channel #';
        READEQID = 'Reading equipment ID.';
   READINGBWLIST = 'Reading bandwidths list.';
 READINGCHANLIST = 'Reading channel list';
 READINGCORSQLEV = 'Reading COR/Squelch level.';
  READINGFIRMVER = 'Reading firmware version.';
  READINGRFAMPST = 'Reading RF amplifier status.';
   READINGIFGAIN = 'Reading IF gain setting.';
 READINGSERPARMS = 'Reading serial parameters.';
    READRXSTATUS = 'Reading RX status.';
       READSERNO = 'Reading RX serial number.';
           READY = 'Ready!';
      REALLYEXIT = 'Really exit?';
          REASON = 'Reason: ';
      RECEIVERID = 'Receiver ID    : ';
   RECEIVERMODEL = 'Receiver model : ';
      RECEIVERSN = 'Receiver SN    : ';
            RECV = 'Recv: ';
     RPLYTIMEOUT = 'Reply timeout!';
           RETRY = 'Retry?';
      RXCANTTUNE = 'RX can''t tune to that frequency.';
      RXREPLYREC = 'Receiver reply received.';
       RXTIMEOUT = 'Timeout waiting for CTS (RX).';
    SCANDELAYMSG = 'Scan delay';
            SENT = 'Sent: ';
       SERFAULTS = 'Serious faults detected.';
   SERPORTOPENED = 'Serial port opened.';
      SETBFOFREQ = 'Set BFO frequency.';
    STARTERRLIST = '----- Error list -----';
         STARTCH = 'Start channel';
       STARTSCAN = 'Start Scan';
        STATEMGR = 'State manager';
      STEP1HZSEL = '1 Hz step selected.';
     STEP10HZSEL = '10 Hz step selected.';
    STEP100HZSEL = '100 Hz step selected.';
   STEP1000HZSEL = '1000 Hz step selected.';
          STOPCH = 'Stop channel';
        STOPSCAN = 'Stop scan';
          SWRATE = 'Sweep rate';
          TITLES = 'TYPE  NUMBER  POS     fLOW   fHIGH  fOFFSET  BANDWIDTH';
  TUNINGDISABLED = 'Tuning disabled.';
       TXTIMEOUT = 'Timeout waiting for CTS (TX).';
UNSUPPORTEDCONFV = 'WARNING: unsupported config file version %s'
                   + LineEnding+'Trying to read it anyway.';
   USEMOUSEWHEEL = 'Use mouse wheel to set IF Gain or squelch level';
     WAUDIOMUTED = 'WARNING: audio muted!';
     WHENBITEFIN = 'When BITE finishes, press ENTER.';
 YOUCANTPLUGHERE = 'You can''t plug phones here 8-)!';

implementation

{$R *.lfm}

{ TRA3701 }

//
// Auxiliary procedures & functions
//

//
// Calculate CRC16 of msg and return it as 3 ASCII character string
// 4 high bits+32, 6 center bits+32 and 6 lower bits+32. See RA3700
// Maintenance Manual page 5.18.
// The problem is that the RA3700 manual does not specify which CRC
// polynomial is to be used. By pure chanche the first tried was the
// correct one.

function TRA3701.CalcCRC(ms: string): string;

const
  // CRC16 parameters
  CRC16_POLY: word = $A001;
  INITIAL_VALUE: word = 0;

//
// This procedure was downloaded from github/prof7bit
//
procedure CRC16_Update(var CRC: Word; Data: Byte); inline;

var
  i: word;

begin
  CRC := CRC xor (Word(Data));
  for i := 0 to 7 do begin
    if (CRC and 1) <> 0 then
      CRC := (CRC shr 1) xor CRC16_POLY
    else
      CRC := CRC shr 1;
  end;
end;

var
  i: integer;
  iCRC: word;
  s: string[3] = '';

begin
  iCRC := INITIAL_VALUE;
  for i := 1 to Length(ms) do CRC16_Update(iCRC,Byte(ms[i]));

  s := s + Char(((iCRC and %1111000000000000) >> 12) + 32);
  s := s + Char(((iCRC and %0000111111000000) >> 6) + 32);
  s := s + Char((iCRC and %0000000000111111) + 32);

  result := s;

  {$IFDEF TESTCRC}
  MsgToShow := 'msg='''+ms+''', iCRC='''+IntToHex(iCRC)+''', CRC='''+s+'''';
  {$ENDIF}
end;

// Interpolate the RF levels table in the RA3700 Maintenance Manual page 5-30
// RA3701 returned RF Level is an 8 bit unsigned integer, 0=120dBuV, 255=0dBuV
function TRA3701.InterpolateRFLevel(RFLevel: integer): real;

begin
  case RFLevel of
    000..097: result := 120.0+10.0*(097-RFLevel)/98; // 98 values
    098..109: result := 110.0+10.0*(109-RFLevel)/12; // 12 values
    110..120: result := 100.0+10.0*(120-RFLevel)/11; // 11 values
    121..132: result :=  90.0+10.0*(132-RFLevel)/12; // 12 values
    133..144: result :=  80.0+10.0*(144-RFLevel)/12; // 12 values
    145..155: result :=  70.0+10.0*(155-RFLevel)/11; // 11 values
    156..167: result :=  60.0+10.0*(167-RFLevel)/12; // 12 values
    168..179: result :=  50.0+10.0*(179-RFLevel)/12; // 12 values
    180..190: result :=  40.0+10.0*(190-RFLevel)/11; // 11 values
    191..202: result :=  30.0+10.0*(202-RFLevel)/12; // 12 values
    203..214: result :=  20.0+10.0*(214-RFLevel)/12; // 12 values
    215..225: result :=  10.0+10.0*(225-RFLevel)/11; // 11 values
    otherwise result :=  0.0 +10.0*(255-RFLevel)/30; // 30 values
  end;
end;

// Set bandwidth values in BW combo per mode
procedure TRA3701.SetBWCombo;

var i,maxBW,BWindex: integer;
    BW: array[0..15] of TBandwidth;

begin
  // Remove all values
  CBXBW.ItemsEx.Clear;

  case RXState.Mode_RX of
    M_USB,M_ISBU: begin
      maxBW := NumUSBBW-1;
      BW := USBBandwidths;
    end;
    M_LSB, M_ISBL: begin
      maxBW := NumLSBBW-1;
      BW := LSBBandwidths;
      BWIndex := -1;
    end;
    otherwise begin           // SYM Filter for CW, AM and FM
      maxBW := NumSYMBW-1;
      BW := SYMBandwidths;
      BWIndex := -1;
    end;
  end;
  for i := 0 to maxBW do begin
    CBXBW.ItemsEx.AddItem(IntToStr(BW[i].Bandwidth));
  end;

  BWIndex := -1;
  for i := 0 to MaxBW do begin
    // MsgToShow := '"'+CBXBW.ItemsEx[i].Caption+'" "'+IntToStr(RXState.Bandwidth)+'"';
    if CBXBW.ItemsEx[i].Caption = IntToStr(RXState.Bandwidth) then begin
      BWIndex := i;
      BWNum := i;
      break;
    end;
  end;
  CBXBW.ItemIndex := BWIndex;
end;

// Set main form caption
procedure TRA3701.SetMainFormCaption;

begin
  {$IFDEF TEST_USER_INTERFACE}
  RA3701.Caption := PROGNAME + ' - USER INTERFACE TEST MODE!';
  {$ELSE}
  RA3701.Caption := PROGNAME + ' ' + ShortVersion
                 +' - '+CommParms.SerPortName+' / '+IntToStr(CommParms.SerportSpeed);
  if CommParms.NumAddrChars > 0 then
    RA3701.Caption := RA3701.Caption + ' / ' + IntToStr(CurRXAddress);
  {$ENDIF}
end;

// Make control chars visible in command strings
function TRA3701.ShowControlChars(cmd: string): string;

var chp: integer;
    stmp: string = '';
    chtmp: string;

begin
  // Display control chars in command strings (see MSHOWCClick)
  for chp := 1 to length(cmd) do begin
    case cmd[chp] of
      {$IFDEF WIN32}
      // No control chars unicode symbols in Win32 (XP)
      // Show only CR as paragraph symbol
      #$0D: chtmp := '¶';
      otherwise chtmp := cmd[chp];
      {$ELSE}
      #$01..#$1f:  begin
        chtmp := ControlImages[ord(cmd[chp])];
      end;
      otherwise chtmp := cmd[chp];
      {$ENDIF}
    end;
    stmp := stmp+chtmp;
  end;
  result := stmp;
end;

// Get RX frequency displayed as a single string
function TRA3701.GetRXString: string;

begin
 GetRXString := Trim(RXFreq10M.Caption+RXFreq1M.Caption+RXFreq100k.Caption+
                RXFreq10k.Caption+RXFreq1k.Caption+RXFreqDOT.Caption+RXFreq100H.Caption+
                RXFreq10H.Caption+RXFreq1H.Caption);
end;

//Put RX string - Put a string into RX frequency display right-aligned
procedure TRA3701.PutRXString(s: string);

var st: string;

begin
 st := s;
 if Length(st) < 9 then st := RightStr('         '+s,9);
 RXFreq10M.Caption := st[1];
 RXFreq1M.Caption := st[2];
 RXFreq100k.Caption := st[3];
 RXFreq10k.Caption := st[4];
 RXFreq1k.Caption := st[5];
 RXFreqDOT.Caption := st[6];
 RXFreq100H.Caption := st[7];
 RXFreq10H.Caption := st[8];
 RXFreq1H.Caption := st[9];
end;

// Get a 2 char string from Channel number TLabels
function TRA3701.GetChanString: string;

begin
  GetChanString:= LBCHANTENS.Caption + LBCHANUNITS.Caption;
end;

// Put a 2 char string into Channel number TLabels
procedure TRA3701.PutChanString(s:string);

begin
  LBCHANTENS.Caption := s[1];
  LBCHANUNITS.Caption := s[2];
end;

// Get a 2 char string from Address TLabels
function TRA3701.GetAddrString: string;

begin
  GetAddrString :=  LBADDRTENS.Caption + LBADDRUNITS.Caption;
end;

// Put a 2 char string into Address TLabels
procedure TRA3701.PutAddrString(s:string);

begin
  LBADDRTENS.Caption := s[1];
  LBADDRUNITS.Caption := s[2];
end;

// Adjust decimal separator in real number string
// Latest compilers do not accept anymore both separators (. and ,)
function TRA3701.AdjustDecimalSeparator(s: string): string;

var i: integer;

begin
  for i := 1 to Length(s) do
    if s[i]='.' then s[i] := DefaultFormatSettings.DecimalSeparator;
  Result := s;
end;

// Process message queue
procedure TRA3701.DoMessages;
begin
  Application.ProcessMessages;
end;

// Read last autosaved state from disk
procedure TRA3701.LoadLastState;

var StateFile: FILE of TRXState;
    tmpRXState: TRXState;
begin
    AssignFile(StateFile, HomeDir+StateFileName);
    if FileExists(HomeDir+StateFileName) then begin
      try
        Reset(StateFile);
        Read(StateFile, tmpRXState);
        CloseFile(StateFile);
        MsgToShow := LOADINGLASTSTA;
      except
        MsgToShow := CANTREADSTATE;
        exit;
      end;
      RXState := tmpRXState;
    end;
end;

// Save last state for parameters non in QALL or for which no read command exists
procedure TRA3701.SaveLastState;

var StateFile: FILE of TRXState;

begin
     AssignFile(StateFile, HomeDir+StateFileName);
     try
        Rewrite(StateFile);
        Write(StateFile, RXState);
        CloseFile(StateFile);
     except
         MsgToShow := CANTWRITESTATE;
     end;
end;

// Open communications
function TRA3701.OpenRemote: boolean;

var StartTime: TTime;
    Timeout: boolean;

begin
  {$IFDEF TEST_USER_INTERFACE}
  exit(NOERROR);
  {$ENDIF}

  // Check for RA3701 responding and get power-on BITE message or any other
  // pending revertive message
  StartTime := Time;
  repeat
    if SendCommand('','') then begin
        // Check for power-on BITE message
        if Pos('BITE1', Status) > 0 then MsgToShow := GOTPWRONBITE+Status;
      if not GetReply then MsgToShow := NOREPLY;
    end else MsgToShow := COMMANDFAILED;
    Timeout := (Time - StartTime > OPENTIMEOUT);
  until (Status = '') or Timeout;

  if Timeout then
    result := ERROR
  else
    result := NOERROR;
end;

// Close communications
function TRA3701.CloseRemote: boolean;
begin
  if SendCommand(Set_Remote, LOCAL) then begin
    if GetReply then begin
      CloseRemote := TRUE;
    end else begin
      MsgToShow := NOREPLY;
      CloseRemote := FALSE;
    end;
  end else begin
    MsgToShow := COMMANDFAILED;
    CloseRemote := FALSE;
  end;
end;

// Make a GetAck alias to GetReply for clarity
function TRA3701.GetAck: boolean; inline;
begin
  result := GetReply;
end;

// Get ACK or reply to commands
// Since ACKs are only empty replies, this routine handles both
function TRA3701.GetReply: boolean;

var buf: char;
    StartTime: TTime;
    nc: integer;
    res: boolean;
    crc: string[3];
    lcc: byte;
    adr: string[2] = '';
    cts: boolean;
    Timeout: boolean;

begin
  {$IFDEF TEST_USER_INTERFACE}
  exit(true);
  {$ENDIF}
  In_Command := TRUE;

  case CommParms.SerPortProtocol of
    PROTO_NONE: ;
    PROTO_CTR: begin
      {$IFDEF DEBUGRS232}
      if SerGetCTS(SerPort) then
        MsgToShow := 'At GetReply enter: CTS true'
      else
        MsgToShow := 'At GetReply enter: CTS false';
      {$ENDIF}

      // Reenable sending from RX
      SerSetRTS(SerPort, TRUE);

      {$IFDEF DEBUGRS232}
      if SerGetCTS(SerPort) then
        MsgToShow := 'After RTS=true: CTS true'
      else
        MsgToShow := 'After RTS=true: CTS false';
      {$ENDIF}
    end;
    PROTO_RTS: begin
      SerSetRTS(Serport,TRUE);
      // This code does not work, at lest on my RA3701. The manual says that
      // when RTS is asserted you have to wait until CTS goes ON, but it never
      // does.
      if Commparms.WaitCTSinRTSProtocol then begin
        StartTime := TIME;
        repeat
          cts := SerGetCTS(SerPort);
          Timeout := Time - StartTime > REPLYTIMEOUT;
        until cts or Timeout;
        if Timeout then begin
          MsgToShow := RXTIMEOUT;
          res := ERROR;
        end;
      end;
    end;
  end;

  buf := chr(0);
  Status := '';

  repeat
    StartTime := Time;
    repeat
      nc := SerRead(SerPort, buf, 1);
      Timeout := (Time - StartTime) > REPLYTIMEOUT;
    until (nc > 0) or timeout;
    Status := Status + buf;
  until (buf = CR) or Timeout;

  if timeout then res := ERROR else res := NOERROR;

  if MSHOWC.Checked then MsgToShow := RECV+ShowControlChars(Status);

  case CommParms.SerPortProtocol of
    PROTO_NONE: ;
    PROTO_CTR: begin
      {$IFDEF RTSDELAY}
      Sleep(25);
      {$ENDIF}
      SerSetRTS(SerPort, FALSE);
    end;
    PROTO_RTS: begin
      {$IFDEF RTSDELAY}
      Sleep(25);
      {$ENDIF}
      SerSetRTS(SerPort, FALSE);
    end;
  end;

  if res = NOERROR then begin
    // Delete opening LF and ending CR
    Delete(Status,1,1);
    Delete(Status,Length(Status),1);

    // If enabled, store, remove and check lcc char
    // lcc = p10iiioo i. e. a char from 01000000 (@) to 11011111 (ß)
    if CommParms.NumLCCChars > 0 then begin
      lcc := Ord(Status[1]);
      Delete(Status,1,1);
      with LCCBits do begin
        Output_Ready := (lcc and $01) > 0;
        Output_Phase := (lcc and $02) > 0;
        Input_Accept := (lcc and $04) > 0;
        Input_Permit := (lcc and $08) > 0;
        Input_Phase  := (lcc and $10) > 0;
        // bit 5 is always 0 ($20)
        // bit 6 is always 1 ($40)
        Parity := (lcc and $80) > 0;
      end;
      // check (tbd)
    end;

    // If enabled, store, remove and check address
    case CommParms.NumAddrChars of
      1: begin
        adr := Copy(Status,1,1);
        Delete(Status,1,1);
      end;
      2: begin
        adr := Copy(Status,1,2);
        Delete(Status,1,2);
      end;
    end;
    if (CommParms.NumAddrChars > 0) and (StrToInt(adr) <> CurRXAddress) then begin
      MsgToShow := Format(ADDRERR, [StrToInt(adr), CurRXAddress]);
      res := ERROR;
    end;

    // If enabled, store, remove and check CRC
    if Commparms.CRCEnabled then begin
      if Length(Status) > 0 then begin
        crc := Copy(Status,Length(Status)-2);
        Delete(Status,Length(Status)-2,3);
        if CalcCRC(adr+Status) <> crc then begin
          MsgToShow := Format(CRCERR, [crc, CalcCRC(adr+Status)]);
          res := ERROR;
        end;
      end;
    end;
  end;

  Result := res;
  In_Command := FALSE;
end;

// Send a command to RA3701
function TRA3701.SendCommand(cmd, par: string): boolean;

var tmp: string;
    StartTime: TTime;
    nc,len,retry: integer;
    ok,Commandsent: boolean;
    buf: array[0..255] of char;
    lcc: string[1] = '';
    adr: string[2] = '';
    crc: string[3] = '';
    cts, timeout: boolean;

begin
 Retry := 0;
 CommandSent := FALSE;

 repeat
    case CommParms.SerPortProtocol of
      PROTO_NONE: ;
      PROTO_CTR: begin
        {$IFDEF DEBUGRS232}
        if SerGetCTS(SerPort) then
          MsgToShow := 'At SendCommand enter: CTS true'
        else
          MsgToShow := 'At SendCommand enter: CTS false';
        {$ENDIF}

        // Stop RX sending while we send a command
        // RX sending will be reenabled in GetReply
        SerSetRTS(SerPort, FALSE);

        {$IFDEF DEBUGRS232}
        if SerGetCTS(SerPort) then
          MsgToShow := 'After RTS=false: CTS true'
        else
          MsgToShow := 'After RTS=false: CTS false';
        {$ENDIF}
      end;
      PROTO_RTS: begin
        SerSetRTS(SerPort, TRUE);
        // On my RA3701 does not work (see above)
        if CommParms.WaitCTSinRTSProtocol then begin
          StartTime := Time;
          repeat
            cts := SerGetCTS(SerPort);
            timeout := (Time-StartTime > CTSTIMEOUT);
          until cts or timeout;
          if timeout then begin
            MsgToShow := TXTIMEOUT;
          end;
        end;
      end;
    end;

    // Calculate lcc, addr and crc values if required, else
    // leave it blank

    // If required, add Address field
    case CommParms.NumAddrChars of
      1: adr := Format('%1d',[CurRXAddress]);
      2: adr := Format('%2.2d',[CurRXAddress]);
    end;
    //  MsgToShow := 'ADR: '+adr;

    if CommParms.CRCEnabled then begin
      if Length(cmd+par) > 0 then begin
        // Add CRC only if not an empty packet
        crc := CalcCRC(adr+cmd+par);
        {$IFDEF DEBUG_CRC}
        MsgToShow := 'CRC = '+crc;
        {$ENDIF}
      end;
    end;

    // LCC tbd

    Enable_Status := DISABLED;

    {$IFDEF USEPAD}
    tmp := LF+lcc+adr+cmd+par+crc+CR+PAD;
    {$ELSE}
    tmp := LF+lcc+adr+cmd+par+crc+CR;
    {$ENDIF}

    len := Length(tmp);

    {$IFDEF TEST_USER_INTERFACE}
    exit(NOERROR);
    {$ENDIF}

    ok := NOERROR;
    StrPCopy(buf,tmp);

    if MSHOWC.Checked then
      MsgToShow := SENT+ShowControlChars(tmp);

    nc := SerWrite(SerPort,buf,Len);

    if nc <> len then begin
      Ok := ERROR;
      {$IFDEF TEST_SENDCOMMAND}
      MsgToShow := 'SerWrite returned ' + IntToStr(nc);
      {$ENDIF}
    end;

    case CommParms.SerPortProtocol of
      PROTO_NONE: ;
      PROTO_CTR: begin
        SerSetRTS(SerPort, TRUE);
        {$IFDEF RTSDELAY}
        Sleep(100);
        {$ENDIF}
        SerSetRTS(SerPort, FALSE);
      end;
      PROTO_RTS: begin
        Sleep(100);
      end;
    end;

     // Ask ACK for the command just sent sending an empty packet
    tmp := LF+lcc+adr+CR;
    len := Length(tmp);

    ok := NOERROR;
    StrPCopy(buf,tmp);

    nc := SerWrite(SerPort,buf,Len);

    if MSHOWC.Checked then
      MsgToShow := SENT+ShowControlChars(tmp);

    if nc <> len then begin
      Ok := ERROR;
      {$IFDEF TEST_SENDCOMMAND}
      MsgToShow := 'SerWrite returned ' + IntToStr(nc);
      {$ENDIF}
    end;

    // Done
    case CommParms.SerPortProtocol of
      PROTO_NONE: ;
      PROTO_CTR: SerSetRTS(SerPort,FALSE);
      PROTO_RTS: SerSetRTS(SerPort,FALSE);
    end;

    if GetAck then begin
      // Command aknowledged, if lcc enabled then check it, else we are done
      if CommParms.NumLCCChars = 1 then begin
        // check lcc and if ok set CommandSent to TRUE - tbd
      end else begin
 {       case CommParms.SerPortProtocol of
          PROTO_NONE: begin
          end;
          PROTO_CTR: begin
             // tbd
          end;
          PROTO_RTS: begin
            SerSetRTS(SerPort, FALSE);
          end;
        end;}
        CommandSent := TRUE;
      end;
    end else begin
      MsgToShow := NOACK+Status;
      ok := ERROR;
    end;
    retry := retry + 1;
  until CommandSent or (Retry > 5);

  if MSmeter.Checked then Enable_Status := ENABLED;

  case CommParms.SerPortProtocol of
    PROTO_NONE: ;
    PROTO_CTR: SerSetRTS(SerPort, FALSE);
    PROTO_RTS: ;
  end;

  result := ok;
end;

// Get full RX status
procedure TRA3701.GetStatus;

begin
  {$IFDEF TEST_USER_INTERFACE}
  Status := 'ANT01;AGC0,1;BFO-2.12K;B03,02;CORL252;F14175000;FLG1;G252;M5;PASSB03,00;PASSF00.00K;BWL1,00,1,01,2,00,2,01,3,00,3,01,3,02,3,03,3,04,3,05;';
  {$ENDIF}
  In_Command := TRUE;

  // Read all parameters (well, almost all...)
  if SendCommand(Get_All, '') then begin
    if GetReply then begin
      if not ParseStatus then MsgToShow := PARSESTFAILED;
    end else begin
      MsgToShow := NOREPLY;
    end;
  end else MsgToShow := COMMANDFAILED;

  {$IFDEF DEBUG_GETSTATUS}
  MsgToShow := 'Status='+Status;
  {$ENDIF}

  // Read parameters not in QALL

  // Squelch
  if SendCommand(Get_Squelch,'') then begin
    if GetReply then begin
      if not ParseStatus then MsgToShow := PARSESTFAILED;
    end else begin
      MsgToShow := NOREPLY;
    end;
  end else MsgToShow := COMMANDFAILED;

  // Current channel number
  if not (KeyboardState in [STATE_SELCHAN, STATE_STOCHAN]) then begin
    if SendCommand(Get_Chan,'') then begin
      if GetReply then begin
        if not ParseStatus then MsgToShow := PARSESTFAILED;
      end else begin
        MsgToShow := NOREPLY;
      end;
    end else MsgToShow := COMMANDFAILED;
  end;

  // RFAMP status
  if SendCommand(Get_RFAMP,'') then begin
    if GetReply then begin
      if not ParseStatus then MsgToShow := PARSESTFAILED;
    end else begin
      MsgToShow := NOREPLY
    end;
  end else MsgToShow := COMMANDFAILED;

 if not DeltaMode_ON then UpdateDisplay;
 In_Command := FALSE;
end;

// Get signal level
function TRA3701.GetLevel: integer;

var i,j: integer;
    err: boolean = FALSE;

begin
  {$IFDEF TEST_GETLEVEL}
    RXState.RFLevel := Random(255);
    exit;
  {$ENDIF}
  if In_Command then exit;
  In_Command := TRUE;
  if RXstate.MeterMode then begin
    if SendCommand(Get_RFlevel, '') then begin
      repeat
        GetReply;
      until Status <> '';
      if Status <> '' then begin
        {$IFDEF TEST_USER_INTERFACE}
        Status := 'RFL128;';
        {$ENDIF}
        i := pos(RevCmds[RFL], Status);
        j := pos(';', Status);
        if i = 1 then begin
          sr := copy(Status, i+3, j-i-3);
          RXState.RFLevel := StrToInt(sr);
        end;
      end else begin
        MsgToShow := NOREPLY;
        err := TRUE;
      end;
    end else begin
      MsgToShow := COMMANDFAILED;
      err := TRUE;
    end;
  end else begin
    if SendCommand(Get_AFlevel, '') then begin
      repeat
        GetReply;
      until Status <> '';
      if Status <> '' then begin
        i := pos(RevCmds[AFL], Status);
        j := pos(';', Status);
        if i = 1 then begin
          sr := copy(Status, i+3, j-i-3);
          RXState.AFLevel := StrToInt(sr);
        end;
      end;
    end else begin
      MsgToShow := COMMANDFAILED;
      err := TRUE;
    end;

    if RXState.Metermode then
      result := RXState.RFLevel
    else
      result := RXState.AFLevel;
  end;

  if err then begin
    // Error occurred, drop value read, if any
    sr := '';
    RXstate.AFLevel := 255;
    RXstate.RFLevel := 255;
{$IFDEF STOPSMONERR}
    TSmeter.Enabled := FALSE;
    TSMDISP.Enabled := FALSE;
    MSmeter.Checked := FALSE;
    Enable_Status := FALSE;
    MsgToShow := ERRLEVEL;
{$ENDIF}
  end;
  In_Command := FALSE;
end;

// Parse status
function TRA3701.ParseStatus: boolean;

var i,n,cmd,t,p,ft,fn: integer;
  stmp, stmp1, stmp2, stmp3: string;
  res: boolean;

begin
  // Status returned by Get_All command is something like
  // 'ANT01;AGC0,1;BFO-2.12K;B03,02;CORL252;F14175000;FLG1;G252;M5;PASSB03,00;
  //  PASSF00.00K;BWL1,00,1,01,2,00,2,01,3,00,3,01,3,02,3,03,3,04,3,05;'
  // Status returned by other Q commands are a subset of this,
  // except QID and ERR which have their own format
  //
  // Returns error status

  // Parse command responses

  {$IFDEF DEBUG_PARSESTATUS}
  MsgToShow := 'Status='+Status;
  {$ENDIF}
  res := NOERROR;

  // If no revertive commands, exit (command confirmation packet)
  if Status = '' then exit(res);

  // Scan returned string for commands
  t := 1;
  cmd := -1;
  repeat
    stmp := ExtractDelimited(t, Status, [';']);
    // No more commands, exit
    if stmp='' then exit(res);

    // Search command
    for i := Low(RevCmds) to High(RevCmds) do begin
      if Pos(RevCmds[i],stmp) = 1 then begin
        cmd := i;
        break;
      end;
    end;

    // Execute command
    case cmd of
      AFL: begin
        RXState.AFLevel := StrToInt(Copy(stmp,4,3));
      end;
      AGC: begin
        RXState.AGCMode := StrToInt(Copy(stmp,4,1));
        RXState.AGCTimeCon := StrToInt(Copy(stmp,6,1));
      end;
      ANT: begin
        RXState.Ant := StrToInt(Copy(stmp,4));
      end;
      AUDIO: begin
        RXState.Audio := stmp[6]='1';
      end;
      B: begin
        ft := StrToInt(Copy(stmp,2,2));
        fn := StrToInt(Copy(stmp,5,2));
        RXState.BWType := ft;
        RXState.BWNumber := fn;
        for i := 0 to 47 do begin
          if (Bandwidths[i].BWType=ft) and (Bandwidths[i].BWNumber=fn) then break;
        end;
        RXState.Bandwidth := Bandwidths[i].Bandwidth;
      end;
      BCON: begin
      // BCON revertive message is parsed elsewhere, so do nothing.
      // If program goes here, there is something wrong.
        MsgToShow := 'NOT GOOD: Got BCON message in ParseStatus';
      end;
      BFO: begin
        RXState.BFO := Trunc(1000*StrToFloat(AdjustDecimalSeparator(copy(stmp,4,5))));
      end;
      BITE: begin
        // implemented in TBITETimer
      end;
      BWL: begin
        i := 0;
        n := 0;
        Delete(stmp,1,3);  // Delete BWL
        Delete(stmp,Length(stmp),1); // Remove terminating ;
        repeat
          i := i+1;
          stmp1 := ExtractDelimited(i,stmp,[',']);
          if stmp1 <> '' then
             Bandwidths[n].BWType := StrToInt(stmp1)
          else
            Bandwidths[n].BWType := -1;
          i := i + 1;
          stmp1 := ExtractDelimited(i,stmp,[',']);
          if stmp1 <> '' then
            Bandwidths[n].BWNumber := StrToInt(stmp1)
          else
            Bandwidths[n].BWNumber := -1;
          n := n+1;
        until (stmp1='') or (n>47); // 16 bandwidths max, 0..15 per type
        i:=0;
        {$IFDEF BWBUG}
        // Disable last bandwidth
        Bandwidths[n-2].BWtype := -1;
        {$ENDIF}
      end;
       C: begin
        RXState.CORLevel := (stmp[2]='1');
        RXState.ScanInhibit := (stmp[4]='1');
      end;
      CHAN: begin
        LastChan := StrToInt(Copy(stmp,5,2));
      end;
      CORL: begin
        RXState.CORSQLevel := StrToInt(Copy(stmp,5,3));
      end;
      COR: begin
        RXState.COREnabled := (stmp[4]='1');
        RXState.CORHangtime := Trunc(10-0*StrToFloat(AdjustDecimalSeparator(Copy(stmp,6,2))));
      end;
      DIVM: begin
        RXstate.DivMaster := (stmp[5]='1');
      end;
      DLTA: begin
        // not revertive
      end;
      ERR: begin
        stmp1 := ExtractDelimited(1,stmp,[',']);
        Delete(stmp1,1,3);
        stmp2 := ExtractDelimited(2,stmp,[',']);
        stmp3 := ExtractDelimited(3,stmp,[',']);
        if stmp1 = '2' then MsgToShow := Format(COMMANDNOTEX,
            [Copy(stmp2,2,Length(stmp2)-2),Copy(stmp3,2,Length(stmp3)-2)]);
        res := ERROR;
      end;
      F: begin
        RXState.Freq_RX := StrToInt(copy(stmp,2));
      end;
      FLG: begin
        RXChannels[CurCHan].Scanflag := (stmp[4]='1');
      end;
      FAULT: begin
        if stmp[5]='0' then FaultDetected := FALSE else FaultDetected := TRUE;
      end;
      G: begin
        RXstate.IFGain := StrToInt(Copy(stmp,2,3));;
      end;
      ID: begin
        Delete(stmp,1,3);
        p := Pos('"', stmp);
        RXModel := copy(stmp,1,p-1);
        LBRAXXXX.Caption := RXModel;
        Delete(stmp,1,p+2);
        p := Pos('"', stmp);
        RXId := copy(stmp,1,p-1)+' SN '+copy(stmp,p+3,4);
        if Length(RXId) > 32 then begin
          LBHFREC.Caption := Copy(RXId,1,31)+'...';
          LBHFREC.Hint := RXId;
        end else begin
          LBHFREC.Caption := RXId;
          LBHFREC.Hint := '';
        end;
      end;
      LINK: begin
        LinkParms := '';
        Delete(stmp,1,4);
        for i := 1 to Length(stmp) do
          if stmp[i] <> '"' then
            LinkParms :=  LinkParms + stmp[i];
      end;
      M: begin
        RXState.Mode_RX := StrToInt(stmp[2])
      end;
      MUTE: begin
        RXstate.Mute := (stmp[5]='1');;
      end;
      PASSB: begin
        ft := StrToInt(Copy(stmp,6,2));
        fn := StrToInt(Copy(stmp,9,2));
        RXState.PASSBType := ft;
        RXState.PASSBNumber := fn;
      end;
      PASSF: begin
        RXState.PASSBFreq := Trunc(1000*StrToFloat(AdjustDecimalSeparator(copy(stmp,6,5))));
      end;
      REM: begin
        // not revertive
      end;
      RFAMP: begin
        RXState.RFAmp := StrToInt(Copy(stmp,6,1));
      end;
      RFL: begin
        RXstate.RFLevel := StrToInt(copy(stmp, 4, 3));
      end;
      SCAN: begin
        // Handled elsewhere
      end;
      SCCH: begin
        // Handled elsewhere
      end;
      SN: begin
        p := Pos('"',stmp);
        Delete(stmp,1,p);
        p := Pos('"',stmp);
        Delete(stmp,p,1);
        RXSN := stmp;
      end;
      SQU: begin
        RXstate.SquelchEnabled := Status[4]='1';
      end;
      SW: begin
        // Reply is something like SW"P87915","12";
        p := Pos('"',stmp);
        Delete(stmp,1,p);
        p := Pos('"',stmp);
        Delete(stmp,p,1);
        FWVersion := Copy(stmp,1,p-1)+'/';
        Delete(stmp,1,p);
        p := Pos('"',stmp);
        Delete(stmp,p,1);
        p := Pos('"',stmp);
        Delete(stmp,p,1);
        FWVersion := FWVersion+stmp;
      end;
      otherwise begin
      MsgToShow := COMMANDNOTFOUND+stmp;
      res := ERROR;
      end;
    end;

    // Next token
    t := t+1;
  until FALSE;
  Result := res;
 end;

// Display update procedures

// Update left panel
procedure TRA3701.UpdateLeftPanel;

var s: string;

begin
  // Make all annunciators invisible except frequency, MASTER
  // and NET which is handled elsewhere
  LBADDR.Hide;
  LBCHAN.Hide;
  LBADDRTENS.Hide;
  LBADDRUNITS.Hide;
  LBCHANTENS.Hide;
  LBCHANUNITS.Hide;
  LBSCAN.Hide;
  LBREMOTE.Hide;
  LBWARN.Hide;

  // Set annunciators to suit RXState

  // show address if selected
  if KeyboardState = STATE_SHOWADDR then begin
    LBADDR.Show;
    LBADDRTENS.Show;
    LBADDRUNITS.Show;
    s := IntToStr(CurRXAddress);
    if CurRXAddress < 10 then s := ' '+s;
    PutAddrString(s);
  end;

  // show channel number and chan flag if in channel mode
  if KeyboardState in [STATE_SELCHAN, STATE_STOCHAN] then begin
    LBWARN.Show;
    LBCHAN.Show;
    LBCHANTENS.Show;
    LBCHANUNITS.Show;
    if RXChannels[CurChan].Scanflag then begin
      LBSCAN.Caption := 'SCAN';
      LBSCAN.Show;
    end;
  end;

  if KeyboardState = STATE_SCAN then begin
    LBCHAN.Show;
    LBCHANTENS.Show;
    LBCHANUNITS.Show;
    LBSCAN.Show;
    LBSCAN.Caption := 'SCANNING';
  end;

  // FAULT
  if FaultDetected then
    LBFAULT.Show
  else
    LBFAULT.Hide;

  // Frequency
  DispRXF;
  DoMessages;
end;

// Format right display line 1 (mode, AGC and squelch)
procedure TRA3701.FormatDisplayLine1(mode,agcm,agct,sq:string);

var stmp:string;

begin
  if Length(mode) = 2 then mode := mode+' ';
  if agcm='' then
    FmtStr(stmp,'%4s       %5s  %2s',[mode,agct,sq])
  else if agct= '' then
    FmtStr(stmp,'%4s      %5s    %2s',[mode,agcm,sq])
  else
    FmtStr(stmp,'%4s    %5s %3s %2s',[mode,agct,agcm,sq]);
  LBDISPLINE1.Caption := stmp;
end;

// Format right display line 2 (BW, BFO & ANT)
procedure TRA3701.FormatDisplayLine2(bw,bfo,ant:string);

var stmp:string;

begin
  // display contains 20 characters
  if RXState.Mode_RX <> M_CW then bfo := '';
  FmtStr(stmp,'%7s %8s %3s',[bw,bfo,ant]);
  LBDISPLINE2.Caption := stmp;
end;

// Update Center panel
procedure TRA3701.UpdateCenterPanel;

begin
  // Hide all labels
  LBAF.Hide;
  LBdbMW1.Hide;
  LBdbMW2.Hide;
  LBdbMW3.Hide;
  LBdbMW4.Hide;
  LBdbMW5.Hide;
  LBDBM.Hide;
  LBRF.Hide;
  LBdb0.Hide;
  LBdb20.Hide;
  LBdb40.Hide;
  LBdb60.Hide;
  LBdb80.Hide;
  LBdb100.Hide;
  LBdb120.Hide;
  LBDBuV.Hide;
  LBMUTE.Hide;
  LBCOR.Hide;
  LBTUNE.Hide;
  LBCHANNEL.Hide;
  LBBFO.Hide;
  LBVARIABLE.Hide;
  LBSLOW.Hide;
  LBMEDIUM.Hide;
  LBFAST.Hide;
  LBLOCK.Hide;

  // Set required RF/AF Meter labels
  if RXState.MeterMode then begin
    LBRF.Show;
    LBdb0.Show;
    LBdb20.Show;
    LBdb40.Show;
    LBdb60.Show;
    LBdb80.Show;
    LBdb100.Show;
    LBdb120.Show;
    LBDBuV.Show;
  end else begin
    LBAF.Show;
    LBdbMW1.Show;
    LBdbMW2.Show;
    LBdbMW3.Show;
    LBdbMW4.Show;
    LBdbMW5.Show;
    LBDBM.Visible := TRUE;
  end;

  // MUTE
  if RXState.Mute then LBMUTE.Visible := TRUE;

  // COR tbd
  //

  // TUNE
  if RXState.Tune then begin
    LBLOCK.Hide;
    case RXState.Freq_step of
      1: begin
        LBSLOW.Visible := TRUE;
        LBSLOW.Caption := 'SLOW+'
      end;
      10: begin
        LBSLOW.Visible := TRUE;
        LBSLOW.Caption := 'SLOW'
      end;
      100: LBMEDIUM.Visible := TRUE;
      otherwise LBFAST.Visible := TRUE;
    end;
  end else
    LBLOCK.Visible := TRUE;


  // CHANNEL tbd
  //

  // BFO is done elsewhere


  // VARIABLE tune rate tbd
  //
  //

  // SLOW tune rate tbd
  //

  // MEDIUM tune rate tbd
  //

  // FAST tune rate tbd
  //

  //
  DoMessages;
end;

// Update Right panel
procedure TRA3701.UpdateRightPanel;

var smode,sagcm,sagct,sbw,sbfo,sant,spre,ssq: string;

begin
  // When menus are shown, hide fixed captions
  if KeyboardState = STATE_MENU then begin
    LBMODE.Hide;
    LBAGC.Hide;
    LBBW.Hide;
    LBBFOF.Hide;
    LBANT.Hide;
    LBMODE.Hide;
  end else begin
    LBMODE.Show;
    LBAGC.Show;
    LBBW.Show;
    LBBFOF.Show;
    LBANT.Show;
    // If mode=CW then show BFO label, else hide it
    if RXState.Mode_RX = M_CW then
      LBBFOF.Visible := TRUE
    else
      LBBFOF.Hide;
  end;

  // Update first line (Mode and AGC)
  case RXState.Mode_RX of
      M_AUX: smode := 'AUX';
      M_USB: smode := 'USB';
      M_LSB: smode := 'LSB';
      M_AM: smode := 'AM';
      M_FM: smode := 'FM';
      M_CW: smode := 'CW';
      M_FSK: smode := 'FSK';
      M_ISBU: smode := 'ISB/U';
      M_ISBL: smode := 'ISB/L';
      otherwise smode := '???';
  end;

  // AGC time constant
  case RXState.AGCTimeCon of
    AGC_SHORT: sagct:= 'SHORT';
    AGC_MEDIUM: sagct := 'MED';
    AGC_LONG: sagct := 'LONG';
    AGC_L2DATA: sagct := 'LK11d';
    AGC_L2NORMAL: sagct := 'LK11n';
    otherwise sagct := '???'
  end;

  // AGC mode
  case RXState.AGCMode of
    AGCON: begin
      sagcm := '';
    end;
    MANONLY: begin
      sagct := '';
      sagcm := 'MAN';
    end;
    MANT: sagcm := 'MAN';
    otherwise sagcm := '???';
  end;

  // Squelch
  SSQ := '';
  if RXState.SquelchEnabled and (RXstate.AGCMode = AGCON) then
    ssq := 'SQ';

  FormatDisplayLine1(smode,sagcm,sagct,ssq);

  // Update second line (BW, BFO, Ant)

  // BFO
  Str((RXState.BFO/1000.0):4:2,sbfo);
  sbfo := sbfo+'KHz';
  if RXState.BFO > 0 then sbfo := '+'+sbfo;

  // ANT
  if RXState.RFAmp > 0 then spre := 'A ' else spre := '  ';
  sant := spre+IntToStr(RXState.Ant);

  // BW
  if RXState.Bandwidth < 10000 then
    Str((RXState.Bandwidth/1000.0):4:2,sbw)
  else
    Str((RXState.Bandwidth/1000.0):4:1,sbw);
  sbw := sbw +'KHz';
  FormatDisplayLine2(sbw,sbfo,sant);
  DoMessages;
end;

// Update IF Gain knob
procedure TRA3701.UpdateIFGainAndSquelch;

var delta,dmax,dstart,R: real;
    xc,yc, dotTop, dotLeft, value: integer;

begin
  // Consts
  dmax := 5.497787144; // in radians, 315°
  dstart := 1.963495408; // in radians, 112.5°
  xc :=  SHRFG.Left+(SHRFG.Width div 2);
  yc := SHRFG.Top+(SHRFG.Height div 2);
  R := 0.35*SHRFG.Width;

  // Update IF GAIN knob position
  if RXState.SquelchEnabled then
    value := RXState.CORSQLevel
  else
    value := RXState.IFGain;

  delta := dstart+dmax*(value)/255.0;
    dotLeft := Round(R*cos(delta))+xc-(SHIFG.Width div 2);
    dotTop := Round(R*sin(delta))+yc-(SHIFG.Width div 2);

  SHIFG.Top := dotTop;
  SHIFG.Left := dotLeft;

  if RXstate.SquelchEnabled then begin
    LBIFGSQ.Caption := IntToStr(RXState.CORSQLevel);
    LBIFGSQ.Show;
  end else begin
    if RXstate.AGCMode in [MANONLY, MANT] then begin
      LBIFGSQ.Caption := IntToStr(RXState.IFGain);
      LBIFGSQ.Show;
    end else LBIFGSQ.Hide;
  end;
  DoMessages;
end;

// Volume knob has no remote command to read/write its setting

// Update Spinedit and combobox, less BW
procedure TRA3701.UpdateSpinAndCombos;

var cbxindex: integer;

begin
  // Frequency combobox
  SpinEditUpdateOnly(SPFREQ, RXState.Freq_rx);
  if RXState.Tune then
    SPFREQ.Enabled := TRUE
  else
    SPFREQ.Enabled := FALSE;

  // Mode combobox
  CBXMode.ItemIndex := RXState.Mode_RX;

  // AGC ComboBox
  if RXState.AGCMode <> 1 then
    cbxindex := 3*RXState.AGCMode + (RXState.AGCTimeCon)
  else
    cbxindex := 5;
  CBXAGC.ItemIndex := cbxindex;
  DoMessages;
end;

// If squelch enabled, change the IF GAIN knob label to "SQ LEVEL"
procedure TRA3701.UpdateSquelchLabel;

begin
  if RXState.SquelchEnabled then
    LBIFG.Caption := 'SQ LEVEL'
  else
    LBIFG.Caption := 'IF GAIN';
  DoMessages;
end;

// Update complete display calling all of the above
procedure TRA3701.UpdateDisplay;

begin
  UpdateLeftPanel;
  UpdateCenterPanel;
  UpdateRightPanel;
  UpdateIFGainAndSquelch;
  UpdateSquelchLabel;
  UpdateSpinAndCombos;
  if RXState.Tune then begin
    if not MTIMED.Checked then begin
      LBKHZT.Caption := IntToStr(HzPerTurn div 1000)+KHZT;
      LBKHZT.Show;
    end else LBKHZT.Hide;
  end else LBKHZT.Hide;
  DoMessages;
end;

// Set display colors
procedure TRA3701.SetColors(AColor: TColor);

var i: integer;

begin
    LBADDR.Font.Color := AColor;
    LBADDRTENS.Font.Color := AColor;
    LBADDRUNITS.Font.Color := AColor;
    LBAF.Font.Color := AColor;
    LBAGC.Font.Color := AColor;
    LBANT.Font.Color := AColor;
    LBBFO.Font.Color := AColor;
    LBBFO.Font.Color := AColor;
    LBBFOF.Font.Color := AColor;
    LBBW.Font.Color := AColor;
    LBBW.Font.Color := AColor;
    LBCHAN.Font.Color := AColor;
    LBCHANNEL.Font.Color := AColor;
    LBCOR.Font.Color := AColor;
    LBCOR.Font.Color := AColor;
    LBdb0.Font.Color := AColor;
    LBdb20.Font.Color := AColor;
    LBdb40.Font.Color := AColor;
    LBdb60.Font.Color := AColor;
    LBdb80.Font.Color := AColor;
    LBdb100.Font.Color := AColor;
    LBdb120.Font.Color := AColor;
    LBDBM.Font.Color := AColor;
    LBDBuV.Font.Color := AColor;
    LBdbMW1.Font.Color := AColor;
    LBdbMW2.Font.Color := AColor;
    LBdbMW3.Font.Color := AColor;
    LBdbMW4.Font.Color := AColor;
    LBdbMW5.Font.Color := AColor;
    LBFAST.Font.Color := AColor;
    LBFAULT.Font.Color := AColor;
    LBFREQ.Font.Color := AColor;
    LBFREQ.Font.Color := AColor;
    LBBITE.Font.Color := clWhite;
    LBKHZ.Font.Color := AColor;
    LBKHZ.Font.Color := AColor;
    LBLOCK.Font.Color := AColor;
    LBMASTER.Font.Color := AColor;
    LBMEDIUM.Font.Color := AColor;
    LBMODE.Font.Color := AColor;
    LBMUTE.Font.Color := AColor;
    LBNETW.Font.Color := AColor;
    LBREMOTE.Font.Color := AColor;
    LBRF.Font.Color := AColor;
    LBSCAN.Font.Color := AColor;
    LBSLOW.Font.Color := AColor;
    LBTUNE.Font.Color := AColor;
    LBVARIABLE.Font.Color := AColor;
    LBWARN.Font.Color := AColor;
    LBDISPLINE1.Font.Color := AColor;
    LBDISPLINE2.Font.Color := AColor;
    LBHFREC.Font.Color := clWhite;
    LBRAXXXX.Font.Color := clWhite;
    RXFreq100H.Font.Color := AColor;
    RXFreq100k.Font.Color := AColor;
    RXFreq10H.Font.Color := AColor;
    RXFreq10k.Font.Color := AColor;
    RXFreq10M.Font.Color := AColor;
    RXFreq1H.Font.Color := AColor;
    RXFreq1k.Font.Color := AColor;
    RXFreq1M.Font.Color := AColor;
    RXFreqDOT.Font.COlor := AColor;

    LSCALE1.Font.Color := AColor;
    LSCALE2.Font.Color := AColor;
    LSCALE4.Font.Color := AColor;
    LSCALE3.Font.Color := AColor;
    LSCALE5.Font.Color := AColor;
    LSCALE6.Font.Color := AColor;
    LSCALE7.Font.Color := AColor;
    LSCALE8.Font.Color := AColor;
    LSCALE9.Font.Color := AColor;
    LSCALE10.Font.Color := AColor;
    LSCALE11.Font.Color := AColor;
    LSCALE12.Font.Color := AColor;
    LSCALE13.Font.Color := AColor;

    for i:=1 to 12 do begin
      (SMeter[i] as TShape).Brush.Color := AColor;
      (SMeter[i] as TShape).Pen.Color := AColor;
    end;
    DoMessages;
end;

// Magnify fonts by the selected factor
procedure TRA3701.FontMagnify;

var c: TControl;
    i, NewHeight1, NewHeight2, NewHeight: integer;

begin
    {$IFNDEF DISABLE_FONTSCALE}
    for i := 0 to ComponentCount-1 do begin
      if (Components[i] is TLabel) or
         (Components[i] is TStaticText) or
         (Components[i] is TMemo) or
         (Components[i] is TJButton) then begin
        {$IFDEF DEBUG_AUTOSCALE}
        MsgToShow := 'scaling font';
        {$ENDIF}
        c := TControl(Components[i]);
        if (c.Tag = 0) or MMAGALLF.Checked then begin
          NewHeight1 := Hf[i] * ClientWidth * FontMagn div (OW * 100);
          NewHeight2 := Hf[i] * ClientHeight * FontMagn div (OH * 100);
          NewHeight := min(abs(NewHeight1), abs(NewHeight2));
        end else begin
          NewHeight1 := Hf[i] * ClientWidth div OW;
          NewHeight2 := Hf[i] * ClientHeight div OH;
          NewHeight := min(abs(NewHeight1), abs(NewHeight2));
        end;
        if (Components[i] is TJButton) then begin
          {$IFDEF DEBUG_FONTSCALE}
          MsgToShow := 'OW='+inttostr(OW)+' OH='+inttostr(OH);
          MsgToShow := 'ClientWidth='+IntToSTr(CLientWidth)+', ClientHeight='+IntToStr(ClientHeight);
          MsgToShow := 'Width='+IntToSTr(Width)+', Height='+IntToStr(Height);
          MsgToShow := 'NewHeight='+IntToStr(NewHeight);
          {$ENDIF}
          (Components[i] as TJButton).LCaption.Font.Height := NewHeight;
        end else begin
          c.Font.Height := NewHeight;
        end
      end;
    end;
    {$ENDIF}
end;

// Set RX frequency
function TRA3701.SetRXFreq: boolean;

var  sf: string;
     res: boolean;

begin
  res := ERROR;
  sf := IntToStr(RXState.Freq_rx);
  if SendCommand(Set_Frequency, sf) then begin
    if GetReply then begin
      OldRXState := RXState;
      res := NOERROR;
    end else MsgToShow := NOACK;
  end else MsgToShow := COMMANDFAILED;
  Result := res;
end;

// Display a message in MSG memo
procedure TRA3701.Message(ms: string);

var LineEndingLen, LineLen: integer;
    stmp: string;

begin
  stmp := ms;
  LineEndingLen := Length(LineEnding);
  LineLen := Pos(LineEnding,stmp);
  if LineLen <> 0 then begin
    // Multiline string, divide it into component lines
    // and add them to the MSG memo
    repeat
      LineLen := Pos(LineEnding,stmp);
      if LineLen=0 then
          LineLen := Length(stmp)
      else
          LineLen := LineLen-1;
      MSG.Lines.Add(copy(stmp,1,LineLen));
      Delete(stmp,1,LineLen+LineEndingLen);
    until Length(stmp) = 0;
  end else begin
      // Single line string
      MSG.Lines.Add(stmp);
  end;
  // kludge to make TMemo scroll - not always works but better than nothing
  MSG.VertScrollBar.Position := 10000;
  DoMessages;
end;

// Clear messages in MSG memo
procedure TRA3701.MSGDblClick(Sender: TObject);
begin
  MSG.Lines.Clear;
end;

// Display only RX frequency
procedure TRA3701.DispRXf;

var tmp: real;
    s: string;

begin
  tmp := RXState.Freq_rx / 1000;
  str(tmp:9:3,s);
  PutRXString(s);
  SpinEditUpdateOnly(SPFREQ, RXState.Freq_RX);
  DoMessages;
end;

// Update SpinEdit display only
procedure TRA3701.SpinEditUpdateOnly(Sender: TObject; value: integer);

var ASpin: TSpinEditEx;

begin
  ASpin := Sender as TSpinEditEx;
  Update_Only := TRUE;
  ASpin.Value := Value;
  Update_Only := FALSE;
end;

// Keyboard handler
procedure TRA3701.KeyboardHandler(keyp: string);

var p: integer;
    s: string;
    f: longint;

begin
  case KeyboardState of
    STATE_NORMAL: {Nothing to do};
    STATE_SETFREQ: begin
      s := GetRXString;
      p := Pos('-',s);
      if p > 0 then begin
        s[p] := keyp[1];
        if s[p+1] = '.' then p := p+2 else p := p+1;
        f := trunc(1000*StrToFloat(AdjustDecimalSeparator(s)));
        if f > MAXRXF then begin
          MsgToShow := RXCANTTUNE;
          exit;
        end;
        if p <= Length(s) then s[p] := '-';
          PutRXString(s);
      end;
    end;
    STATE_SELCHAN, STATE_STOCHAN: begin
      sChtmp[1]:= sChtmp[2];
      sChtmp[2] := keyp[1];
      PutChanString(sCHtmp);
    end;
    STATE_SETADDR: begin
      case CommParms.NumAddrChars of
        1: begin
          sChtmp[1] := ' ';
          if keyp[1] in ['0'..'9'] then begin
             sChtmp[2] := keyp[1];
          end;
          keyp := '';
       end;
        2: begin
          sChtmp[1]:= sChtmp[2];
          if keyp[1] in ['0'..'9'] then begin
            sChtmp[2] := keyp[1];
          end;
          keyp := '';
        end;
      end;
      PutAddrString(sChtmp);
    end;
  end;
end;

// Save RX channels file to disk
procedure TRA3701.SaveChanFile;

var RXChanFile: file of TRXChannels;

begin
  AssignFile(RXChanFile, HomeDir+RXChanFileName);
  try
    Rewrite(RXChanFile);
    Write(RXChanFile, RXChannels);
    CloseFile(RXChanFile);
  except
    MsgToShow := CANTWRITECHAN;
  end;
end;

// Read RX channels file from disk
procedure TRA3701.ReadChanFile;

var RXChanFile: file of TRXChannels;

begin
  AssignFile(RXChanFile, HomeDir+RXChanFileName);
  try
    Reset(RXChanFile);
    Read(RXChanFile, RXChannels);
    CloseFile(RXChanFile);
  except
    MsgToShow := NOCHANFILE;
  end;
end;

// Save config to disk - XMLconfig version
procedure TRA3701.SaveConfig;

var
  Config: TXMLConfig;
  w,h,l,t,i: integer;
  s: string;

begin
    // Create the TXMLconfig object
    Config := TXMLConfig.Create(self);
    Config.Filename := Homedir+XMLConfigFileName;
    Config.RootName := 'RA3701ControlConfig';

    // Start with an empty file
    Config.Clear;

    // Save config version number
    Config.SetValue('ConfigVersion','1');

    // Save communications parameters
    with CommParms do begin
      Config.OpenKey('CommsSetup');
      Config.SetValue('SerialPortName', UniCodeString(SerPortName));
      Config.SetValue('SerialPortSpeed', SerPortSpeed);
      Config.SetValue('SerialPortParity', SerPortParity);
      Config.SetValue('SerialPortProtocol', SerPortProtocol);
      Config.SetValue('WaitCTSinRTSProtocol', WaitCTSinRTSProtocol);
      Config.SetValue('NumLCCChars', NumLCCChars);
      Config.SetValue('NumAddrChars', NumAddrChars);
      Config.SetValue('CRCEnabled', CRCEnabled);
      if NumRX = 0 then begin
        msgtoshow := 'NumRX = 0 ???';
        NumRX := 1;
      end;
      Config.SetValue('NumRX', NumRX);
      if CommParms.NumAddrChars > 0 then begin
        for i := 1 to NumRX do begin
          s := Format('RX%dAddress', [i]);
          Config.SetValue(UniCodeString(s), RXaddresses[i]);
        end;
      end;
      Config.SetValue('CurrentRXNumber', CurRXNum);
      Config.CloseKey;
    end;

    // Save program config parameters
    Config.OpenKey('ProgramConfig');
    Config.SetValue('ShowMeter', MSMETER.Checked);
    Config.SetValue('MeterMode', RXState.MeterMode);
    Config.SetValue('ShowHamlibCmds', MSHOWHAMLIB.Checked);
    Config.SetValue('ShowRXCmds',MSHOWC.Checked);
    Config.SetValue('EnableAllCmds', MENALL.Checked);
    Config.SetValue('ContinuousBITE', MCONTBITE.Checked);
    Config.SetValue('LCDColor', UniCodeString(sLEDColor));
    Config.SetValue('HzPerTurn', HzPerTurn);
    Config.SetValue('UseProgramChannels', MUSEPROGCH.Checked);
    Config.SetValue('FontMagnification', FontMagn);
    Config.SetValue('MagnifyAllFonts', MMAGALLF.Checked);
    Config.SetValue('LockRXInRemote', MLOCKRX.Checked);
    Config.SetValue('RFAmplifier', RXstate.RFAmp);
    Config.SetValue('RestoreStateAfterScan', MRESTSTATE.Checked);

    w := ClientWidth;
    h := ClientHeight;
    l := Left;
    t := Top;

    // Check w, h, l and t values. Under XP and maybe some other system
    // Top and Left may go negative, so check them also
    if w > Screen.Width then w := Screen.Width;
    if h > Screen.Height then h := Screen.Height;
    if t < 0 then t := 0;
    if l < 0 then l := 0;

    {$IFDEF DEBUG_SAVEDSIZE&POS}
    MsgToShow := Format('Saved size&pos: %dx%d@%d,%d',[w, h, l,t]);
    DoMessages;
    sleep(5000);
    {$ENDIF}

    Config.SetValue('MainWindowWidth', w);
    Config.SetValue('MainWindowHeight', h);
    Config.SetValue('MainWindowLeft', l);
    Config.SetValue('MainWindowTop', t);
    Config.SetValue('TCPServerHost', UniCodeString(RA3701Server.Host));
    Config.SetValue('TCPServerPort', RA3701Server.Port);
    Config.CloseKey;
    Config.Flush;
    Config.Free;
    SetMainFormCaption;
end;

// Read config from disk
procedure TRA3701.ReadConfig;
var s,ConfVersion: string;
    Config: TXMLConfig;
    i: integer;

begin
  s := HomeDir+XMLConfigFileName;
  if FileExists(s) then begin
  // We have an XML config file, read it
    try
      Config := TXMLConfig.Create(self);
      Config.Filename := s;
      ConfVersion := AnsiString(Config.GetValue('ConfigVersion',''));

      if ConfVersion <> '1' then begin
        FmtStr(s, UNSUPPORTEDCONFV, [ConfVersion]);
        MsgToShow := s;
      end;

      // Serial communications setup
      with CommParms do begin
        Config.OpenKey('CommsSetup');
        SerPortName := AnsiString(Config.GetValue('SerialPortName','ttyS0'));
        SerPortSpeed := Config.GetValue('SerialPorSpeed', 9600);
        SerPortParity := Config.GetValue('SerialPortParity', 0);
        SerPortProtocol := Config.GetValue('SerialPortProtocol', PROTO_NONE);
        WaitCTSinRTSProtocol := Config.GetValue('WaitCTSinRTSProtocol', FALSE);
        NumLCCChars := Config.GetValue('NumLCCChars', 0);
        NumAddrChars := Config.GetValue('NumAddrChars', 33);
        CRCEnabled := Config.GetValue('CRCEnabled', FALSE);
        NumRX := Config.GetValue('NumRX', 1);

        if (NumAddrChars > 0) then begin
          for i := 1 to NumRX do begin
            s := Format('RX%dAddress', [i]);
            RXaddresses[i] := Config.GetValue(UniCodeString(s), i);
          end;
          SPRXNUM.MaxValue := NumRX;

        end;
        RA3701.LBRXNUM.Caption := 'RX#';
        RA3701.SPRXNUM.MinValue := 1;
        RA3701.SPRXNUM.MaxValue := NumRX;
        CurRXNum := Config.GetValue('CurrentRXNumber', 1);
        if CurRXNum > NumRX then CurRXNum := 1;
        CurRXAddress := RXAddresses[CurRXNum];
        SPRXNUM.Value := CurRXNum;
        Config.CloseKey;
      end;

      Config.OpenKey('ProgramConfig');
      MSMETER.Checked := Config.GetValue('ShowMeter', FALSE);
      RXState.MeterMode := Config.GetValue('MeterMode', FALSE);
      MSHOWHAMLIB.Checked := Config.GetValue('ShowHamlibCmds', FALSE);
      MSHOWC.Checked := Config.GetValue('ShowRXCmds', FALSE);
      MENALL.Checked := Config.GetValue('EnableAllCmds', FALSE);
      MCONTBITE.Checked := Config.GetValue('ContinuousBITE', TRUE);
      sLEDColor := AnsiString(Config.GetValue('LCDColor', 'yellow'));
      HzPerTurn := Config.GetValue('HzPerTurn', 1000);
      MUSEPROGCH.Checked := Config.GetValue('UseProgramChannels', FALSE);
      MRESTSTATE.Checked := Config.GetValue('RestoreStateAfterScan',FALSE);
      FontMagn := Config.GetValue('FontMagnification', 100);
      MMAGALLF.Checked := Config.GetValue('MagnifyAllFonts', FALSE);
      MLOCKRX.Checked := Config.GetValue('LockRXInRemote', TRUE);
      RXstate.RFAmp := Config.GetValue('RFAmplifier', 0);
      StartWidth := Config.GetValue('MainWindowWidth', 1077);
      StartHeight := Config.GetValue('MainWindowHeight', 371);
      StartX := Config.GetValue('MainWindowLeft', 0);
      StartY := Config.GetValue('MainWindowTop', 0);
      RA3701Server.Host := AnsiString(Config.GetValue('TCPServerHost', '127.0.0.1'));
      RA3701Server.Port := Config.GetValue('TCPServerPort', 4538);
      Config.CloseKey;
      Config.Free;

      if sLEDColor = 'black' then begin
        LEDColorON := clBlack;
        MBLACK.Checked := TRUE;
      end else if sLEDColor = 'blue' then begin
        LEDColorON := clBlue;
        MBLUE.Checked := TRUE;
      end else if sLEDColor = 'yellow' then begin
        LEDColorON := DispForeColor;
        MYELLOW.Checked := TRUE;
      end else begin
        // color name invalid, use black
        LEDColorON := clBlack;
        MBLACK.Checked := TRUE;
      end;
      SetColors(LEDColorOn);

      case HzPerTurn of
        0:      MTIMED.Checked := TRUE;
        500:    M0P5KHZT.Checked := TRUE;
        1000:   M1KHZT.Checked := TRUE;
        10000:  M10KHZT.Checked := TRUE;
        100000: M100KHZT.Checked := TRUE;
        otherwise begin
          HzPerTurn := 1000;
          M1KHZT.Checked := TRUE;
        end;
      end;

      case FontMagn of
        100: MMAG100.Checked := TRUE;
        110: MMAG110.Checked := TRUE;
        125: MMAG125.Checked := TRUE;
        150: MMAG150.Checked := TRUE;
        175: MMAG175.Checked := TRUE;
        200: MMAG200.Checked := TRUE;
      end;

      case RXstate.RFAmp of
        0: MRFAMPOFF.Checked := TRUE;
        1: MRFAMPON.Checked := TRUE;
        2: MRFAMPAUTO.Checked := TRUE;
      end;
    except
      ShowMessage(ERRORREADINGCONF);
    end;
  end else begin
    // No configuration file, set up and save a default one
    ShowMessage(CONFIGNOTFOUND);
    SetDefaultConfig;
    SaveConfig;
  end;
  MRXADD.Caption := Format(MRXADDCAPT,[CurRXAddress]);
end;

// Set a default configuration if no config file found
procedure TRA3701.SetDefaultConfig;

begin
  // default values
  {$IFDEF LINUX}
  CommParms.SerPortName := '/dev/ttyS0';
  {$ENDIF}

  {$IFDEF WINDOWS}
  CommParms.SerPortName := '\\.\COM1';
  {$ENDIF}
  CommParms.SerPortSpeed := 9600;
  MSmeter.Checked := TRUE;
  CommParms.CRCEnabled := FALSE;
  CommParms.NumAddrChars := 0;
  CommParms.NumLCCChars := 0;
  CommParms.NumRX := 1;
  CommParms.RXAddresses[1] := 1;
  CommParms.SerPortParity := PARITY_EVEN;
  CommParms.CRCEnabled := FALSE;
  CommParms.SerPortProtocol := PROTO_CTR;
  CurRXNum := 1;
  CurRXAddress := 1;
  MRFAMPOFF.Checked := TRUE;
  LEDColorOFF := $ff740a;
  LEDColorON := clYellow;
  sLEDColor := 'yellow';
  MYellow.Checked := TRUE;
  MBlack.Checked := FALSE;
  MBlue.Checked := FALSE;
  MMAG100.Checked := TRUE;
  MLOCKRX.Checked := TRUE;
  MUSEPROGCH.Checked := FALSE;
  M10KHZT.Checked := TRUE;
  RXState.MeterMode := TRUE;
  RXState.RFAmp := 0;

  // Default window placement if no config file
  StartWidth :=  1109;
  StartHeight := 320;
  StartX := (Screen.Width-StartWidth) div 2;
  StartY := (Screen.Height - StartHeight) div 2;
end;

// Restore state in RXState
procedure TRA3701.RestoreState;

var par: string;

begin
  // Set RX frequency
  SpinEditUpdateOnly(SPFREQ, RXstate.Freq_RX);
  if SendCommand(Set_Frequency, IntToStr(RXState.Freq_RX)) then begin
    if not GetReply then MsgToShow := NOREPLY;
  end else begin
    MsgToShow := COMMANDFAILED;
  end;

  // Set mode
  if RXState.Mode_RX in [M_AUX..M_ISBL] then begin
    par := IntToStr(RXState.Mode_RX);
  end else begin
    // Invalid value, default to A1
    RXState.Mode_RX := M_CW;
    par := '5';
  end;
  if SendCommand(Set_Mode, par) then begin
    if not GetReply then MsgToShow := NOREPLY;
  end else begin
    MsgToShow := COMMANDFAILED;
  end;

  // Update Bandwidths for new mode
  SetBWCombo;

  // If in CW mode, set BFO
  if RXstate.Mode_RX = M_CW then begin
    if SendCommand(Set_BFO, IntToStr(RXState.BFO)) then begin
      if not GetReply then MsgToShow := NOREPLY;
    end else begin
      MsgToShow := COMMANDFAILED;
    end;
  end;

  // Set bandwidth
  FmtStr(par,'%1D,%0.2D', [RXState.BWType,RXState.BWNumber]);
  if SendCommand(Set_IFBW, par) then begin
    if not GetReply then MsgToShow := NOREPLY;
  end else begin
    MsgToShow := COMMANDFAILED;
  end;

  // Set AGC
    FmtStr(par,'%1D,%0.2D', [RXState.AGCMode,RXState.AGCTimeCon]);
    if SendCommand(SET_AGC, par) then begin
      if not GetReply then MsgToShow := NOREPLY
    end else begin
      MsgToShow := COMMANDFAILED;
    end;

  // Status polling
  if MSmeter.Checked then begin
    TSmeter.Enabled := TRUE;
  end else begin
    TSmeter.Enabled := FALSE;
  end;

  // Scan mode
  if RXState.Scan then begin
    //TODO?
  end;

  // TUNE
  if RXstate.Tune then begin
    LBLOCK.Hide;
    SPFREQ.Enabled := TRUE;
  end else begin
    LBLOCK.Show; //Visible := TRUE;
    SPFREQ.Enabled := FALSE;
  end;

  // Squelch does not works in MAN modes
  if RXState.SquelchEnabled then begin
    if (RXState.AGCMode <> AGCON) then begin
      MsgToShow := NOMANIFGSQ;
      DoMessages;
    end;
    // Set squelch
    if SendCommand(Set_Squelch, '1') then begin
      if not GetReply then MsgToShow := NOREPLY;
    end else begin
      MsgToShow := COMMANDFAILED;
    end;
  end else begin // squelch off
    if SendCommand(Set_Squelch, '0') then begin
      if not GetReply then MsgToShow := NOREPLY;
    end else begin
      MsgToShow := COMMANDFAILED;
    end;
  end;

  // Get and display new parameters
  In_Command := FALSE;
  GetStatus;
  if not ParseStatus then MsgToShow := PARSESTFAILED;
  UpdateDisplay;
  SetActiveButtons;
  UpdateSpinAndCombos;
end;

// Create ordered Channels submenus
procedure TRA3701.LoadChannelsDir;

var m: TMenuItem;
    n: integer;
    Info: TSearchRec;
    FileList: TStringList;

begin
  FileList := TStringList.Create;
  FileList.Sorted := TRUE;
  if FindFirst(ChannelsDir+'*.dat', faAnyFile, info) = 0 then begin
    repeat
      delete(Info.Name, Length(Info.Name)-3, 4);
      FileList.Add(Info.Name);
    until FindNext(Info) <> 0;
    FindClose(Info);
    MChannels.Clear;
    for n := 0 to FileList.Count-1 do begin
      m := TmenuItem.Create(MChannels);
      m.Caption := FileList.Strings[n];
      m.OnClick := @LoadChannel;
      MChannels.Add(m);
    end;
    FileList.Clear;
    FileList.Free;
  end else begin
    MChannels.Clear;
    m := TmenuItem.Create(MChannels);
    m.Caption := NOCHANDEF;
    MChannels.Add(m);
  end;
end;

// Load a channel file
procedure TRA3701.LoadChannel(Sender: TObject);

var BandFile: FILE of TRXState;
    tmpRXState: TRXState;

begin
  In_Command := TRUE;
  AssignFile(BandFile,ChannelsDir+(Sender as TMenuItem).Caption+'.dat');
  MsgToShow := LOADINGCH;
  try
    Reset(BandFile);
    Read(BandFile,tmpRXState);
    CloseFile(BandFile);
  except
    MsgToShow := CANTOPENCHAN;
    exit;
  end;
  RXstate := tmpRXState;
  RestoreState;
  UpdateDisplay;
  In_Command := FALSE;
  MsgToShow := DONE;
end;

// Disable all controls except REM and some Menus
procedure TRA3701.DisableAllControls;

var i: integer;

begin
  for i := 0 to ComponentCount - 1 do begin
    if (Components[i] is TLabel) then begin
      if (Components[i] = LBHFREC) or
         (Components[i] = LBRAXXXX1) or
         (Components[i] = LBBITE) or
         (Components[i] = LBIFG) or
         (Components[i] = LBVOLUME) or
         (Components[i] = LBPOWER) or
         (Components[i] = LBPHN) then continue;
      TLabel(Components[i]).Hide;
    end;
    if (Components[i] is TControl) then
      TControl(Components[i]).Enabled := FALSE;
  end;

  // Hide SMeter shapes
  for i := 1 to 12 do SMeter[i].Hide;

  // Disable Menus requiring connection with the receiver...
  MLState.Enabled := FALSE;
  MSState.Enabled := FALSE;
  MSCHANN.Enabled := FALSE;
  MRSTATE.Enabled := FALSE;
  MSAVEAUX.Enabled := FALSE;
  MSmeter.Enabled := FALSE;
  MChannels.Enabled := FALSE;
  MOTHER.Enabled := FALSE;
  MRFAMP.Enabled := FALSE;
  MCONTBITE.Enabled := FALSE;

  //...and enable those who don't
  MSERPARMS.Enabled := TRUE;
  MRXADD.Enabled := TRUE;
  BREM.Enabled := TRUE;
  RXFreq1H.Enabled := TRUE;
  RXFreq10H.Enabled := TRUE;
  RXFreq100H.Enabled := TRUE;
  RXFreqDOT.Enabled := TRUE;
  RXFreq1k.Enabled := TRUE;
  RXFreq10k.Enabled := TRUE;
  RXFreq100k.Enabled := TRUE;
  RXFreq1M.Enabled := TRUE;
  RXFreq10M.Enabled := TRUE;
  LBADDR.Enabled := TRUE;

  // Enable ON/OFF switch
  SHSW.Enabled := TRUE;
  SHSWb2.Enabled := TRUE;
end;

// Disable all buttons
procedure TRA3701.DisableAllButtons;

var i: integer;

begin
  if MENALL.Checked then exit;
  for i := 0 to ComponentCount - 1 do begin
    if Components[i] is TJButton then
      TJButton(Components[i]).Enabled := FALSE;
    if Components[i] is TSpinEditEx then
      TSpinEditEx(Components[i]).Enabled := FALSE;
    if Components[i] is TComboBoxEx then
      TComboBoxEx(Components[i]).Enabled := FALSE;
  end;
end;

// Disable numeric keypad buttons
procedure TRA3701.DisableKeypad;

begin
  if MENALL.Checked then exit;
  B0.Enabled := FALSE;
  B1.Enabled := FALSE;
  B2.Enabled := FALSE;
  B3.Enabled := FALSE;
  B4.Enabled := FALSE;
  B5.Enabled := FALSE;
  B6.Enabled := FALSE;
  B7.Enabled := FALSE;
  B8.Enabled := FALSE;
  B9.Enabled := FALSE;
  BPLMI.Enabled := FALSE;
end;

// Enable all buttons
procedure TRA3701.EnableAllButtons;

var i: integer;

begin
  for i := 0 to ComponentCount - 1 do begin
     if Components[i] is TJButton then
       TJButton(Components[i]).Enabled := TRUE;
  end;
end;

// Enable numeric keypad buttons
procedure TRA3701.EnableKeypad;

begin
  B0.Enabled := TRUE;
  B1.Enabled := TRUE;
  B2.Enabled := TRUE;
  B3.Enabled := TRUE;
  B4.Enabled := TRUE;
  B5.Enabled := TRUE;
  B6.Enabled := TRUE;
  B7.Enabled := TRUE;
  B8.Enabled := TRUE;
  B9.Enabled := TRUE;
  BPLMI.Enabled := TRUE;
end;

// Enable command keys
procedure TRA3701.EnableCmdKeys;

begin
  BREM.Enabled := TRUE;
  BRCL.Enabled := TRUE;
  BBFO.Enabled := TRUE;
  BUSB.Enabled := TRUE;
  BCHAN.Enabled := TRUE;
  BFREQ.Enabled := TRUE;
  BISB.Enabled := TRUE;
  BLSB.Enabled := TRUE;
  BFSK.Enabled := TRUE;
  BAM.Enabled := TRUE;
  BCW.Enabled := TRUE;
  BFM.Enabled := TRUE;
  BTUNEP.Enabled := TRUE;
  BTUNEM.Enabled := TRUE;
  BSCAN.Enabled := TRUE;
  BSTORE.Enabled := TRUE;
end;

// Enable Combo and Edit boxes
procedure TRA3701.EnableCombosAndEdits;
begin
  SPFREQ.Enabled := TRUE;
  CBXMODE.Enabled := TRUE;
  CBXBW.Enabled := TRUE;
  CBXAGC.Enabled := TRUE;
end;

// Enable all controls
procedure TRA3701.EnableAllControls;

var i: integer;

begin
  for i := 0 to ComponentCount - 1 do begin
    if Components[i] is TMenuItem then begin
        TMenuItem(Components[i]).Enabled := TRUE;
    end;
    if Components[i] is TSpinEditEx then begin
        TSpinEditEx(Components[i]).Enabled := TRUE;
    end;
    if Components[i] is TComboBoxEx then begin
        TComboBoxEx(Components[i]).Enabled := TRUE;
    end;
    if Components[i] is TShape then begin
        TShape(Components[i]).Visible := TRUE;
        TShape(Components[i]).Enabled := TRUE;
    end;
    if Components[i] is TPanel then begin
      TPanel(Components[i]).Enabled := TRUE;
      TPanel(Components[i]).Color := DispBackColor;
    end;

    if (Components[i] is TLabel) then begin
      if (Components[i].Tag <> -1) then begin
        TLabel(Components[i]).Font.Color := LEDColorON;
        TLabel(Components[i]).Color := DispBackColor;
      end;
      TLabel(Components[i]).Enabled := TRUE;
      TLabel(Components[i]).Show;
    end;
  end;
  if MSmeter.Checked then begin
    TSmeter.Enabled := TRUE;
  end else begin
    TSmeter.Enabled := FALSE;
    for i := 1 to 12 do (Smeter[i] as TShape).Hide;
  end;
  MSG.Enabled := TRUE;
  DoMessages;
end;

// Set active/not active buttons & edit state
procedure TRA3701.SetActiveButtons;

begin
  if MENALL.Checked then
      // Do not try to disable inactive buttons, so do nothing and return
      exit
  else begin
    case KeyboardState of
      STATE_NORMAL: begin
        EnableAllButtons;
        EnableKeypad;
        if not RXState.Tune then SPFREQ.Enabled := FALSE;
      end;
      STATE_SETFREQ: begin
        DisableAllButtons;
        EnableKeypad;
        BRCL.Enabled := TRUE;
        BFREQ.Enabled := TRUE;
        BENTER.Enabled := TRUE;
      end;
      STATE_SELCHAN: begin
        DisableAllButtons;
        EnableKeypad;
        BRCL.Enabled := TRUE;
        BSCAN.Enabled := TRUE;
        BENTER.Enabled := TRUE;
      end;
      STATE_STOCHAN: begin
        DisableAllButtons;
        EnableKeypad;
        BENTER.Enabled := TRUE;
      end;
      STATE_SETBFO: begin
        DisableAllButtons;
        BBFO.Enabled := TRUE;
        BENTER.Enabled := TRUE;
        BREM.Enabled := TRUE;
      end;
      STATE_SCAN: begin
        BREM.Enabled := FALSE;
        BBFO.Enabled := FALSE;
        BTUNEP.Enabled := FALSE;
        BSTORE.Enabled := FALSE;
        BENTER.Enabled := FALSE;
        BFREQ.Enabled := FALSE;
        BCHAN.Enabled := FALSE;
        BRCL.Enabled := FALSE;
        BISB.Enabled := FALSE;
        BLSB.Enabled := FALSE;
        BUSB.Enabled := FALSE;
        BAM.Enabled := FALSE;
        BCW.Enabled := FALSE;
        BFM.Enabled := FALSE;
        BSCAN.Enabled := TRUE;
      end;
      STATE_BITEFACT,STATE_BITECONF,STATE_BITEFF: begin
        DisableAllButtons;
        BENTER.Enabled := TRUE;
        BRCL.Enabled := TRUE;
        MsgToShow := WHENBITEFIN;
      end;
    end;
  end;
end;


//
// Object init, procedures and functions
//

// Various initializations at startup
procedure TRA3701.FormCreate(Sender: TObject);

var i: integer;

begin
  // If an error occurred during initialization phase (ErrMsg not empty)
  // display an error window and terminate program now
  if ErrMsg <> '' then begin
    MessageDlg(ERRORMSG, ErrMsg, mtError,[mbOK],'');
    halt;
  end;

  // Under my Win10 the font is rendered too big with
  // the default height of 20 so decrese it.
  // At 19 it is a little too small, but can't do otherwise.
  {$IFDEF WINDOWS}
  LBDISPLINE1.Font.Height := 19;
  LBDISPLINE2.Font.Height := 19;
  {$ENDIF}

  // Clear display lines
  LBDISPLINE1.Caption := '';
  LBDISPLINE2.Caption := '';

  {$IFDEF TEST_USER_INTERFACE}
  RA3701.Caption := 'RA3701 Control - USER INTERFACE TEST MODE!';
  {$ELSE}
  RA3701.Caption := 'RA3701 Control ' + ShortVersion;
  {$ENDIF}

  // Initialize SMeter array
     SMeter[1] := SHB1;
     Smeter[2] := SHB2;
     Smeter[3] := SHB3;
     Smeter[4] := SHB4;
     Smeter[5] := SHB5;
     Smeter[6] := SHB6;
     Smeter[7] := SHB7;
     Smeter[8] := SHB8;
     Smeter[9] := SHB9;
     Smeter[10] := SHB10;
     Smeter[11] := SHB11;
     SMeter[12] := SHB12;

  // Check X and Y PPI
  PPI_X := Graphics.ScreenInfo.PixelsPerInchX;
  PPI_Y := Graphics.ScreenInfo.PixelsPerInchY;
  if PPI_X <> PPI_Y then
    ShowMessage('This screen does not have square pixels:'+
    LineEnding+'PPI_X='+IntTostr(PPI_X)+' PPI_Y='+IntTostr(PPI_Y)+
    LineEnding+'Expect strange behaviour in scaling routines.');

  PutRXString('');
  BREM.Enabled := TRUE;
  MSmeter.Enabled := FALSE;
  MOTHER.Enabled := FALSE;
  TBITE.Enabled := FALSE;
  KeyboardState := STATE_POWEROFF;

  // Read channels directory
  LoadChannelsDir;

  // Read config
  ReadConfig;

  // Calculate fixed parameters for tuning knob
  r := (SHKNOB1.Width - SHKNOB.Width) / 2;
  cx := SHKNOB1.Left+(SHKNOB1.Width - SHKNOB.Width) / 2;
  cy := SHKNOB1.Top+(SHKNOB1.Height -SHKNOB.Height) / 2;
  X2 := SHKNOB1.Width / 2;
  Y2 := SHKNOB1.Height / 2;
  R2 := SHKNOB1.Width / 2;

  SPFREQ.Enabled := FALSE;

  // Component & window scaling
  {$IFDEF AUTOSCALE}
  for i := 0 to ComponentCount - 1 do
    if Components[i] is TControl then begin
      Xs[i] := TControl(Components[i]).Left;
      Ys[i] := TControl(Components[i]).Top;
      Ws[i] := TControl(Components[i]).Width;
      Hs[i] := TControl(Components[i]).Height;
    if (Components[i] is TLabel) or
       (Components[i] is TMemo) or
       (Components[i] is TSpinEditEx) or
       (Components[i] is TComboBoxEx) then begin
      Hf[i] := TControl(Components[i]).Font.Height;
    end;
    if (Components[i] is TJButton) then begin
      Hf[i] := (Components[i] as TJButton).LCaption.Font.Height;
    end;
  end;
  {$IFDEF DEBUG_AUTOSCALE}
  MsgToShow := 'total of '+IntToStr(i)+' components';
  {$ENDIF}

  // Save start width and height
  OH := ClientHeight;
  OW := ClientWidth;

  {$IFDEF DEBUG_AUTOSCALE}
  MsgToShow := 'H='+IntTostr(Height)+' W='+IntTostr(Width);
  MsgToShow := 'OH='+IntToStr(OH)+' OW='+IntToSTR(OW);
  {$ENDIF}
  {$ENDIF}
end;

// Close down program
procedure TRA3701.CloseProgram;
begin
  if MSmeter.Checked then begin
    TSmeter.Enabled := FALSE;
    Enable_Status := FALSE;
  end;
  if SerPort > 0 then begin
    ReleaseKBLock;
    SerFlushInput(SerPort);
    SerFlushOutput(SerPort);
    SerClose(SerPort);
  end;
  Halt;
end;

// Closing by clicking on X
procedure TRA3701.FormClose(Sender: TObject; var CloseAction: TCloseAction);
begin
  CloseAction := caNone;
  MExitClick(Sender);
end;

// Release RX KB lock
procedure TRA3701.ReleaseKBLock;

begin
  if not SendCommand(Set_Remote,REMOTE_NOKBLOCK) then MsgToShow := COMMANDFAILED;
end;

// Allows to enter commands using the keyboard
procedure TRA3701.FormKeyPress(Sender: TObject; var Key: char);

var f: integer;

begin
  if SPFREQ.Focused or In_ChanUpdate then exit;

  if KeyboardState = STATE_SETADDR then begin
    KeyboardHandler(Key);
    exit;
  end;

  {$IFDEF TEST_KEYPRESS}
  MsgToShow := 'Key pressed: '+''''+Key+'''';
  {$ENDIF}

  case LowerCase(Key) of
    'u': BUSBCLick(Sender);
    'l': BLSBClick(Sender);
    'c': BCWClick(Sender);
    'a': BAMClick(Sender);
    'f': if RXState.Mode_RX = M_FM then
           BFSKClick(Sender)
         else
           BFMClick(Sender);
    'i': BISBClick(Sender);
    'b': begin
     // To be written
      RestoreState;
      UpdateDisplay;
    end;
    'r': begin
       BRCLClick(Sender);
    end;
    '0': B0Click(Sender);
    '1': B1Click(Sender);
    '2': B2Click(Sender);
    '3': B3Click(Sender);
    '4': B4Click(Sender);
    '5': B5Click(Sender);
    '6': B6Click(Sender);
    '7': B7Click(Sender);
    '8': B8Click(Sender);
    '9': B9Click(Sender);
    '*': begin
      if RXState.Tune then
        BFREQCLick(Sender)
      else
        MsgToShow := TUNINGDISABLED;
    end;
    '.': BSQLClick(Sender);
    '/': begin
      case RXState.Freq_step of
         1: RXState.Freq_step := 10;
        10: RXState.Freq_step := 100;
        100: RXState.Freq_step := 1000;
        1000: RXState.Freq_step := -1;
        otherwise RXState.Freq_step := 1;
      end;
      if RXstate.Freq_Step = -1 then
        RXState.Tune := FALSE
      else
        RXState.Tune := TRUE;
      UpdateDisplay;
    end;
    #$0D: BENTERCLick(Sender);
    '+': begin
      if RXState.Tune then begin
        f := RXState.Freq_rx + RXState.Freq_step;
        if f <= MAXRXF then begin
          RXState.Freq_rx := f;
          SetRXFreq;
          UpdateDisplay;
        end;
      end;
    end;
    '-': begin
      if RXstate.Tune then begin
        f := RXstate.Freq_rx - RXState.Freq_step;
        if f >= MINRXF then begin
          RXState.Freq_rx := f;
          SetRXFreq;
          UpdateDisplay;
        end;
      end;
    end;
  end;
  key := #0;
end;

// Perform BITE at specified level
procedure TRA3701.DoBITE(BITELevel: integer);

const Doomsday = FALSE;

var stest, sprevtest, sLevelDescr,sresult,sErrNum, sSeverity,
    sBlock, sDescr: string;
    sFFTestNum, sFFTestBlock, sFFTestDescr,sFFTestNumSave: string;
    pBITE,PFault,i: integer;
    ErrorsDetected: boolean = FALSE;
    FaultStatus: string = '';

begin
  OldRXState := RXState;
  // Save Smeter and continuous BITE timers status and disable them
  OldTSen := TSMeter.Enabled;
  OldTBEn := TBITE.Enabled;
  TSmeter.Enabled := FALSE;
  TBITE.Enabled := FALSE;

  FEntTestNo.Show;
  repeat
    DoMessages;
  until OKPressed;
  OKPressed := FALSE;
  FEntTestNo.Hide;

  DisableAllButtons;
  BRCL.Enabled := TRUE;

  In_Command := TRUE;
  RCLPressed := FALSE;
  slevel := IntToStr(BITELevel);
  case BITELevel of
    2: sLevelDescr := CONFBITE;
    4: sLevelDescr := FFBITE;
    5: sLevelDescr := FACTBITE;
    otherwise sLevelDescr := Format(BITELEVDESCR,[BITElevel]);
  end;
  sErrNum := '';

  FmtStr(sTest, '%3.3d',[StrToInt(FEntTestNo.EDTestNo.Text)]);
  FmtStr(sPrevTest, '%3.3d',[StrToInt(FEntTestNo.EDTestNo.Text)]);
  {$IFDEF DEBUG_BITE}
  msgtoshow := 'Entering loop...';
  {$ENDIF}
  if SendCommand(Set_BITE,slevel+','+sTest) then begin
    {$IFDEF DEBUG_BITE}
    msgtoshow := 'After Sendcommand: '+ Status;
    {$ENDIF}
    Sleep(1000);  // wait receiver setup for BITE
    if GetReply then begin
      {$IFDEF DEBUG_BITE}
      msgtoshow := 'Reply:'+Status;
      {$ENDIF}
    end else MsgToShow := NOREPLY;
    DisableAllButtons;
    BRCL.Enabled := TRUE;
    repeat
      if stest <> '999' then begin
        LBBITE.Font.Color := clWhite;
        LBBITE.Font.Style := [];
        GetTestDescription(stest,sBlock,sDescr);
        LBBITE.Caption := Format(BITELABFORMAT,[sLevelDescr,stest,sBlock,sDescr,'']);
        DoMessages;
      end;
      Sleep(500);
      repeat
        if SendCommand(Get_BITE,'') then begin
          // Check for FAULT reply.
          // Manual says it must come back only revertively as response
          // to QFAULT command, but my RA3701 sends it here
          pFault := Pos('FAULT', Status);
          if pFault > 0 then FaultStatus := Status;
          if GetReply then begin
            {$IFDEF DEBUG_BITE}
            msgtoshow := 'After Get_BITE: ' + Status;
            {$ENDIF}
            // Check for valid BITE response,
            // e.g. BITE2,0123,message; or BITE2,0123;
            // and check also for FAULT message if not found before.
            pBITE := Pos('BITE', Status);
            if pFault = 0 then begin
              pFault := Pos('FAULT', Status);
              if pFault > 0 then FaultStatus := Status;
            end;
            if pBITE > 0 then begin
              if (Status[pBITE+4] in ['0'..'9']) and
                 (Status[pBITE+5] = ',') and
                 (Status[pBITE+10] in[';',',']) then begin
                stest := Copy(Status,pBITE+6,4);
                if sTest[1]='0' then Delete(sTest,1,1);
                  if Status[pBITE+10] = ',' then
                    sResult := Copy(Status,11)
                  else
                    sResult := '';
              end;
            end;
          end else MsgToShow := NOREPLY;
        end else MsgToShow := COMMANDFAILED;
        for i := 1 to 10 do begin
          Sleep(50);
          DoMessages;
        end;
        {$IFDEF DEBUG_BITE}
        msgtoshow := 'At end of loop: '+ stest+' '+sprevtest;
        {$ENDIF}
      until (stest <> sprevtest) or (sResult <> '') or (PFault > 0) or RCLPressed;

      // Check for failed test
      if pFault > 0 then begin
        LBBITE.Font.Color := clRED;
        LBBITE.Font.Style := [fsBold];
        sSeverity := Copy(FaultStatus,6,1);
        GetTestDescription(sTest,sBlock,sDescr);
        // Save data of the first failed test
        sFFTestNum := Copy(FaultStatus,pFAULT+10, Pos(';', Status)-8);
        sFFTestNumSave := sFFTestNum;
        sFFTestBlock := sBlock;
        sFFTestDescr := sDescr;
        ErrorsDetected := TRUE;
        LBBITE.Caption := Format(BITELABFORMAT,[sLevelDescr,stest,sBlock,sDescr,FAILED]);
        MsgToShow := LBBITE.Caption;
        MsgToShow := PRESSRCLTOCONT;
        KeyboardState := STATE_BITESTOP;
        repeat
          DoMessages;
        until RCLPressed;
        RCLPressed := FALSE;
      end else begin
        LBBITE.Font.Color := clWhite;
        LBBITE.Font.Style := [];
        LBBITE.Caption := LBBITE.Caption + PASSED;
      end;
      DoMessages;

      if stest <> '999' then begin
        sPrevtest := stest;
        if stest <> '' then snexttest := IntToStr(StrToInt(stest)+1);
        {$IFDEF NOTDEF}
        // Manual says so, but my RA3701 behaves otherwise
        if sresult <> '' then begin
          // If result not empty, there was a fault, at least according to manual
          LBBITE.Font.Color := clRED;
          LBBITE.Font.Style := [fsBold];
          sErrNum := BITEFAILED+Copy(sResult,12,Pos(';',sResult)-12);
          GetTestDescription(stest,sBlock,sDescr);
          LBBITE.Caption := Format(BITELABFORMAT,[sLevelDescr,stest,sBlock,sDescr,FAILED]);
          MsgToShow := LBBITE.Caption;
          MsgToShow := PRESSRCLTOCONT;
          KeyboardState := STATE_BITESTOP;
        end else begin
          LBBITE.Font.Color := clWhite;
          LBBITE.Font.Style := [];
          GetTestDescription(stest,sBlock,sDescr);
          LBBITE.Caption := Format(BITELABFORMAT,[sLevelDescr,stest,sBlock,sDescr,PASSED]);
        end;
        {$ENDIF}
      end else begin // BITE end
        if ErrorsDetected then begin
          LBBITE.Font.Color := clRED;
          LBBITE.Font.Style := [fsBold];
          LBBITE.Caption := Format(BITECOMPLFAULT,[slevel, sFFTestNum, sFFTestBlock, sFFTestDescr]);
        end else begin
          LBBITE.Font.Color := clWhite;
          LBBITE.Font.Style := [];
          LBBITE.Caption := Format(BITECOMPLNOFAULT,[slevel]);
        end;
        MsgToshow := PRESSRCLTOTERM;
        KeyboardState := STATE_BITEEND;
        In_Command := FALSE;
        exit;
      end;
      for i := 1 to 10 do begin
        sleep(10);
        DoMessages;
      end;
      if RCLPressed then begin
        RCLPressed := FALSE;
        TSMeter.Enabled := OldTSen;
        TBITE.Enabled := OldTBEn;
        LBBITE.Caption := '';
        break;
      end;
    until Doomsday;
  end else MsgToShow := COMMANDFAILED;
end;

// Main form resize
procedure TRA3701.FormResize(Sender: TObject);

var i: integer;
    c: TControl;
    NewHeight, NewHeight1, NewHeight2: integer;
    alpha, myx, myy: real;

begin
  {$IFDEF AUTOSCALE}
  In_Command := TRUE; // do not start a command while resizing
  { Here's actual component scaling. }
  for i := 0 to ComponentCount - 1 do begin
    if Components[i] is TControl then begin
      c := TControl(Components[i]);
      c.Left   := Xs[i] * ClientWidth div OW;
      c.Top    := Ys[i] * ClientHeight div OH;
      c.Width  := Ws[i] * ClientWidth div OW;
      c.Height := Hs[i] * ClientHeight div OH;
      if c is TComboBoxEx then
        (c as TComboBoxEx).ItemHeight := Hs[i] * ClientHeight div OH;
      {$IFDEF DEBUG_AUTOSCALE}
      MsgToShow := 'h='+IntTostr(c.Height)+' was '+IntTostr(Hs[i]);
      {$ENDIF}
    end;
    c := TControl(Components[i]);
    if (c is TLabel) or
       (c is TJButton) or
       (c is TMemo) or
       (c is TSpinEditEx) or
       (c is TComboBoxEx) or
       (c is TShape) then begin
      {$IFDEF DEBUG_AUTOSCALE}
      MsgToShow := 'scaling font';
      {$ENDIF}
      // Scale using ClientWidth since main screen is
      // more wide than high except for SMeter scale
      NewHeight1 := Hf[i] * ClientWidth div OW;
      NewHeight2 := Hf[i] * ClientHeight div OH;

      if c.Tag = -2 then
        NewHeight := NewHeight2
      else
        NewHeight := NewHeight1;

      if c is TJButton then
        (c as TJButton).LCaption.Font.Height := NewHeight
      else
        c.Font.Height := NewHeight;
    end;
    {$IFDEF DEBUG_AUTOSCALE}
    MsgToShow := 'newheight1='+inttostr(NewHeight1)+' newheight2='+inttostr(NewHeight2);
    MsgToShow := 'newheight='+inttostr(NewHeight);
    MsgToShow := 'Width='+IntToSTr(Width)+', Height='+IntToStr(Height);
    MsgToShow := 'ClientWidth='+IntToSTr(CLientWidth)+', ClientHeight='+IntToStr(ClientHeight);
    {$ENDIF}
  end;

  // Recalculate parameters for tuning knob
  r := (SHKNOB1.Width - SHKNOB.Width) / 2.3;
  cx := SHKNOB1.Left+(SHKNOB1.Width - SHKNOB.Width) / 2;
  cy := SHKNOB1.Top+(SHKNOB1.Height -SHKNOB.Height) / 2;
  X2 := SHKNOB1.Width / 2;
  Y2 := SHKNOB1.Height / 2;
  R2 := SHKNOB1.Width / 2;

  // Put knob in the new position
  alpha := arctan2(SHKNOB1.Top-SHKNOB1.Height / 2, SHKNOB1.Left + SHKNOB1.Width / 2);
  myx := r * cos(alpha) + cx;
  myy := r * sin(alpha) + cy;
  SHKNOB.Top := trunc(myy);
  SHKNOB.Left := trunc(myx);

  // Set IF Gain / Squelch knob
  UpdateIFGainAndSquelch;

  // Put ON/OFF switch in the correct position
  if KeyBoardState <> STATE_POWEROFF then SHSW.Top := SHSWB2.Top;

  {$IFDEF WINDOWS}
  CBXMODE.ClearSelection;
  CBXBW.ClearSelection;
  CBXAGC.ClearSelection;
  UpdateDisplay;
  {$ENDIF}
  {$ENDIF}
  In_Command := FALSE; // done
end;

// Main form things to do when shown
procedure TRA3701.FormShow(Sender: TObject);

var i: integer;

begin
  // Restore saved width & height
  RA3701.Height := StartHeight;
  RA3701.Width := StartWidth;

  // Adapt main window to the screen height/width
  if RA3701.Height > Screen.Height then
    RA3701.Height := Screen.Height;
  if RA3701.Width > Screen.Width then
    RA3701.Width := Screen.Width;

  //Put window in the saved position
  if StartX <= Screen.Width - RA3701.Width then
    RA3701.Left := StartX
  else
    RA3701.Left := Screen.Width - RA3701.Width;

  if StartY <= Screen.Height - RA3701.Height then
    RA3701.Top := StartY
  else
    RA3701.Top := Screen.Height - RA3701.Height;

  {$IFDEF TEST_USER_INTERFACE}
  GetStatus;
  UpdateDisplay;
  EnableCmdKeys;
  EnableCombosAndEdits;
  SPFREQ.Enabled := FALSE;
  {$ENDIF}

  // Set main form caption
  SetMainFormCaption;

  // Disable all controls except ON/OFF switch
  DisableAllControls;

  DoMessages;

  // Hide labels & shapes, will be shown later if required
  for i := 0 to ComponentCount - 1 do begin
    if Components[i] is TLabel then
      TLabel(Components[i]).Visible:= FALSE;
    if Components[i] is TShape then begin
      if Components[i].Tag <> -1 then
        TShape(Components[i]).Visible:= FALSE;
    end;
  end;

  {$IFDEF TESTCRC}
  // Show "Test CRC button"
  MTESTCRC.Show;
  MTESTCRC.Enabled := TRUE;
  {$ENDIF}
end;

//
// Buttons handling
//

//
// Leftmost buttons column
//

// "REM"
procedure TRA3701.BREMClick(Sender: TObject);

begin
  In_Command := TRUE;
  if KeyboardState = STATE_NORMAL then begin
    if Remote_Enabled then begin
      if SendCommand(Set_Remote, LOCAL) then begin
        if GetAck then begin
          Remote_Enabled := FALSE;
          LBREMOTE.Hide;
        end else MsgToShow := NOACK;
      end else MsgToShow := COMMANDFAILED;
    end else begin
      if MLOCKRX.Checked then begin
        if SendCommand(Set_Remote, REMOTE_KBLOCK) then begin
          if GetAck then begin
            Remote_Enabled := TRUE;
            LBREMOTE.Show;
          end else MsgToShow := NOACK;
        end else MsgToShow := COMMANDFAILED;
      end else begin
        if SendCommand(Set_Remote, REMOTE_NOKBLOCK) then begin
          if GetAck then begin
            Remote_Enabled := TRUE;
            LBREMOTE.Show;
          end else MsgToSHow := NOACK;
        end else MsgToShow := COMMANDFAILED;
      end;
    end;
  end;

  // Update RX status if we went in remote from local
  if Remote_Enabled then begin
    GetStatus;
    if not ParseStatus then MsgToShow := PARSESTFAILED;
    UpdateDisplay;
  end;
  In_Command := FALSE;
end;

// "ADDR"
procedure TRA3701.BADDRClick(Sender: TObject);
begin
  case KeyboardState of
    STATE_NORMAL: begin
      KeyboardState := STATE_SHOWADDR;
      UpdateLeftPanel;
    end;
    STATE_SHOWADDR: begin
      KeyboardState := STATE_SETADDR;
      LBWARN.Show;
      PutAddrString(' _');
      MsgToShow := ENTERRXADDR;
    end;
    STATE_SETADDR: begin
      LBWARN.Hide;
      KeyboardState := STATE_SHOWADDR;
      UpdateLeftPanel;
    end;
  end;
end;


// ON/OFF switch - start/stop program
procedure TRA3701.SHSWMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);

var i,p,LastSYMBW: integer;
    stmp,CORL: string;
    Parity: TParityType;

begin
  if KeyboardState = STATE_POWEROFF then begin
    // We were in power off mode, bring up program

    // Toggle simulated switch
    SHSW.Top := SHSWB2.Top;

    KeyboardState := STATE_NORMAL;
    MSG.Lines.Clear;
    ReadConfig;
    ReadChanFile;
    SetMainFormCaption;

    // Try to open serial port
    {$IFDEF TEST_USER_INTERFACE}
    SerPort := 1;
    {$ELSE}
    SerPort := SerOpen(CommParms.SerPortName);
    {$ENDIF}
    if SerPort > 0 then begin
      MsgToShow := SERPORTOPENED;
      case CommParms.SerPortParity of
          0: Parity := NONEParity;
          1: Parity := EVENParity;
          2: Parity := ODDParity;
          otherwise Parity := EVENParity;
      end;
      SerSetParams(SerPort,CommParms.SerPortSpeed,7, Parity ,1,[]);

      // DTR is not used, but set it anyway
      SerSetDTR(Serport, TRUE);

      // Set RTS
      case Commparms.SerPortProtocol of
        PROTO_NONE: SerSetRTS(Serport, TRUE);
        PROTO_CTR: SerSetRTS(Serport, FALSE);
        PROTO_RTS: SerSetRTS(Serport, FALSE);
        otherwise SerSetRTS(Serport, TRUE);
      end;

      TSMETER.Enabled := FALSE;
      TSMDISP.Enabled := FALSE;
      TBITE.Enabled := FALSE;
      Enable_Status := FALSE;
      TSMDISP.Enabled := FALSE;
      MSmeter.Enabled := FALSE;

      // Open communications with the receiver
      if OpenRemote then begin
        {$IFDEF TEST_REMOTE}
        MsgToShow := Status;
        {$ENDIF}
        MsgToShow := RXREPLYREC;
        In_Command := TRUE;

        // Set remote lock mode
        if MLOCKRX.Checked then begin
          if SendCommand(Set_Remote, REMOTE_KBLOCK) then begin
            if not GetReply then MsgToShow := NOREPLY;
          end else MsgToShow := COMMANDFAILED;
        end else begin
          if SendCommand(Set_Remote,REMOTE_NOKBLOCK) then begin
            if not GetReply then MsgToShow := NOREPLY;
          end;
        end;

        // Read receiver serial number
        if SendCommand(Get_SerialNo,'') then begin
          if GetReply then begin
            {$IFDEF TEST_USER_INTERFACE}
            Status := 'SN"0538";';
            {$ENDIF}
            if not ParseStatus then begin
              MsgToShow := PARSESTFAILED;
            end else MsgToShow := READSERNO;
          end else MsgToShow := NOREPLY;
        end else MsgToShow := COMMANDFAILED;

        // Get RX ID
        if SendCommand(Get_EQID,'') then begin
          if GetReply then begin
            {$IFDEF TEST_USER_INTERFACE}
            Status := 'ID"RA3701/7";';
            {$ENDIF}
            if not ParseStatus then
              MsgToShow := PARSESTFAILED
            else
              MsgToShow := READEQID;
          end else MsgToShow := NOREPLY;
        end else MsgToShow := COMMANDFAILED;

        // Read firmware version
        if SendCommand(Get_SWProgNo,'') then begin
          if GetReply then begin
            {$IFDEF TEST_USER_INTERFACE}
            Status := 'SW"P87915","12";';
            {$ENDIF}
            if not ParseStatus then
              MsgToShow := PARSESTFAILED
            else
              MsgToShow := READINGFIRMVER;
          end else MsgToShow := NOREPLY;
        end else MsgToShow := COMMANDFAILED;

         // Issue QLINK command to retrieve serial parameters
        MsgToShow := READINGSERPARMS;
        if SendCommand(Get_LinkParms,'') then begin
          if GetReply then begin
            {$IFDEF TEST_USER_INTERFACE}
            Status := 'LINK"RTS","9600","9600","EVEN","NOLCC","01","CRC","0","0"';
            In_Command := FALSE;
            {$ENDIF}
            if not ParseStatus then MsgToShow := PARSESTFAILED;
          end else MsgToShow := NOREPLY;
        end else MsgToShow := COMMANDFAILED;

       {$IFNDEF TEST_USER_INTERFACE}
       // Read bandwidths list
        if SendCommand(Get_BWList,'') then begin
          if GetReply then begin
            if not ParseStatus then
              MsgToShow := PARSESTFAILED
            else
              MsgToShow := READINGBWLIST;
          end else MsgToShow := NOREPLY;
        end else MsgToShow := COMMANDFAILED;

        // Read bandwidths data
        NumUSBBW := 0;
        NumLSBBW := 0;
        NumSYMBW := 0;
        for i := 0 to 47 do begin
          if Bandwidths[i].BWType < 0 then break;
          if SendCommand(Get_BWConf,
            IntToStr(Bandwidths[i].BWType)+','+
            IntToStr(Bandwidths[i].BWNumber)) then begin
            if GetReply then begin
              {$IFDEF DEBUG_QBCON}
              MsgToShow := Status;
              {$ENDIF}
              // Parse QBCON reply
              // BCON response format:
              //     TY NO PO fLO    fHi    foff   i/r
              // BCON03,01,02,-0.50K,00.50K,00.00K,1;
              //
              {$IFDEF TEST_USER_INTERFACE}
              Status := 'BCON03,01,02,-0.50K,00.50K,00.00K,1';
              {$ENDIF}
              if Status = '' then continue;
              stmp := ExtractDelimited(3,Status,[',']);
              Bandwidths[i].Position := StrToInt(stmp);
              stmp := ExtractDelimited(4,Status,[',']);
              p := Pos('K',stmp);
              if p>0 then Delete(stmp,p,1);
              Bandwidths[i].FLow := Trunc(StrToFloat(AdjustDecimalSeparator(stmp))*1000.0);
              stmp := ExtractDelimited(5,Status,[',']);
              p := Pos('K',stmp);
              if p>0 then Delete(stmp,p,1);
              Bandwidths[i].FHigh := Trunc(StrToFloat(AdjustDecimalSeparator(stmp))*1000.0);
              stmp := ExtractDelimited(6,Status,[',']);
              p := Pos('K',stmp);
              if p>0 then Delete(stmp,p,1);
              Bandwidths[i].FOffset := Trunc(StrToFloat(AdjustDecimalSeparator(stmp))*1000.0);
              Bandwidths[i].Bandwidth := Bandwidths[i].FHigh-Bandwidths[i].fLow;

              if Bandwidths[i].BWType = BWTYPE_USB then begin
                USBBandwidths[NumUSBBW] := Bandwidths[i];
                Inc(NumUSBBW);
              end;
              if Bandwidths[i].BWType = BWTYPE_LSB then begin
                LSBBandwidths[NumLSBBW] := Bandwidths[i];
                Inc(NumLSBBW);
              end;
              if Bandwidths[i].BWType = BWTYPE_SYM then begin
                SYMBandwidths[NumSYMBW] := Bandwidths[i];
                LastSYMBW := i;
                Inc(NumSYMBW);
              end;
              PLEFT.Color := DispBackColor;
              PCENTER.Color := DispBackColor;
              PRIGHT.Color := DispBackColor;
            end else MsgToShow := NOREPLY;
          end else MsgToShow := COMMANDFAILED;
        end;

        // My RA3791 repeats bandwidth 0 of the SYM group at the end of BW list.
        // Maybe a bug in the insert bandwidth routine or in the QBWL one.
        // If all RA3701's have this bug no matter what, then $DEFINE BWBUG in defines.inc
        NumBandwidths := i;
        {$IFDEF BWBUG}
        Bandwidths[LastSYMBW].BWtype := -1;
        Dec(NumBandwidths);
        Dec(i);
        {$ENDIF}
        FmtStr(stmp, READBWLIST, [i]);
        MsgToShow := stmp;

        // Issue QRFAMP command to retrieve RF preamplifier setting
        MsgToShow := READINGRFAMPST;
        if SendCommand(Get_RFAmp,'') then begin
          if GetReply then begin
            if not ParseStatus then MsgToShow := PARSESTFAILED;
          end else MsgToShow := NOREPLY;
        end else MsgToShow := COMMANDFAILED;

        // Issue QG command to retrieve IF gain setting
        MsgToShow := READINGIFGAIN;
        if SendCommand(Get_IFGain,'') then begin
          if GetReply then begin
            if not ParseStatus then MsgToShow := PARSESTFAILED;
          end else MsgToShow := NOREPLY;
        end else MsgToShow := COMMANDFAILED;
        In_Command := FALSE;

        // Retrieve COR/SQUELCH level
        MsgToShow := READINGCORSQLEV;
        if SendCommand(Get_CORSQLevel, '') then begin
          if GetReply then begin
            p := Pos('CORL',Status);
            if p > 0 then begin
              CORL := Copy(Status,p+4,3);
              RXState.CORSQLevel := StrToInt(CORL);
            end;
          end else MsgToShow := NOREPLY;
        end else MsgToShow := COMMANDFAILED;

        // Read last state file
        LoadLastState;

        // Read channel file, if any
        ReadChanFile;

        // Get actual RX status and update display
        MsgToShow := READRXSTATUS;
        GetStatus;
        SetBWCombo;
        {$ELSE}
        Status := 'ID"RA3701/7","HF RECEIVER 100 KHZ IF FSK DEMOD","0538"';
        ParseStatus;
        {$ENDIF}

        // Start
        MsgToShow := READY;

        OldRXState := RXState;
        DoMessages;
        if MSmeter.Checked then begin
          TSmeter.Enabled := TRUE;
          Enable_Status := TRUE;
          TSMDISP.Enabled := TRUE;
        end else begin
          TSmeter.Enabled := FALSE;
          Enable_Status := FALSE;
          TSMDISP.Enabled := FALSE;
          RXState.RFLevel := 255;
          RXState.AFLevel := 255;
        end;
        if MCONTBITE.Checked then
          TBITE.Enabled := TRUE
        else
          TBITE.Enabled := FALSE;
        if RXstate.Tune then begin
          if not MTIMED.Checked then begin
            LBKHZT.Caption := IntToStr(HzPerTurn div 1000)+KHZT;
            LBKHZT.Show;
          end else LBKHZT.Hide;
        end;
        TUPDDISP.Enabled := TRUE;
        FontMagnify;
        KeyboardState := STATE_NORMAL;
        EnableAllControls;
        EnableAllButtons;
        UpdateSpinAndCombos;
        UpdateDisplay;

        // We are the Master
        LBMASTER.Show;

        // Start rigctl server
        if not RA3701Server.Listen(RA3701Server.Port,RA3701Server.Host) then
          MsgToShow := LISTENFAILED;

        // Show/hide the NET label
        if NumTCPConnections > 0 then
          LBNETW.Show
        else
          LBNETW.Hide;
        In_Command := FALSE;
      end else begin  // OpenRemote failed
        SerFlushInput(SerPort);
        SerFlushOutput(SerPort);
        SerClose(SerPort);
        SerPort := 0;
        case MessageDlg(ERRORMSG, FAILEDINITCOMMS+LineEnding+RETRY,
             mtConfirmation, [mbYES, mbNO],0) of
          mrNo: CloseProgram;
          mrYes: begin
            In_Command := FALSE;
            KeyboardState := STATE_POWEROFF;
            SerPort := 0;
            SaveLastState;
            SHSWMouseUp(Sender,mbLeft,[],0,0);
          end;
        end;
      end;
    end else begin   // Serial port opening failed
      if MessageDlg('Error',
         'Unable to open serial port "'
         +CommParms.SerPortname
         +'".'
         +LineEnding
         +'Check name and user privileges.'
         + LineEnding
         +'Retry?',
         mtConfirmation, [mbYES, mbNO],0) = mrNO then begin
        CloseProgram
      end else begin
        In_Command := FALSE;
      end;
    end;
  end else begin  // not KeyboardState=POWEROFF: we are ON, so bring down program
    if In_Command then exit;

    // Unlock RX keyboard
    ReleaseKBLock;

    TSMETER.Enabled := FALSE;
    TSMDISP.Enabled := FALSE;
    TBITE.Enabled := FALSE;
    Enable_Status := FALSE;
    TSMDISP.Enabled := FALSE;
    MSmeter.Enabled := FALSE;
    // Toggle simulated switch
    SHSW.Top := SHSWB2.Top+SHSW.Height;
    CloseRemote;
    SerFlushInput(SerPort);
    SerFlushOutput(SerPort);
    SerSetRTS(SerPort,FALSE);
    SerSetDTR(SerPort, FALSE);
    SerClose(SerPort);
    SerPort := 0;
    SaveConfig;
    SaveChanFile;
    SaveLastState;
    PutRXString('');
    DisableAllControls;

    for i := 0 to ComponentCount - 1 do begin
      if Components[i] is TLabel then
        TLabel(Components[i]).Visible:= FALSE;
      if Components[i] is TShape then begin
        if Components[i].Tag <> -1 then
          TShape(Components[i]).Visible:= FALSE;
      end;
    end;
    CBXMODE.ItemIndex := -1;
    CBXBW.ItemIndex := -1;
    CBXAGC.ItemIndex := -1;
    MBSCANW.Enabled := FALSE;
    RA3701Server.Disconnect(TRUE);
    NumTCPConnections := 0;
    KeyboardState := STATE_POWEROFF;
    In_Command := FALSE;
    In_Wheel := FALSE;
    In_RXFreqMouseUp := FALSE;
    In_TXFreqMouseUp := FALSE;
    In_ChanMouseWheel:= FALSE;
    In_ChanMouseUp := FALSE;
  end;
end;

//
// Center-left buttons block
//

// "FREQ"
procedure TRA3701.BFREQClick(Sender: TObject);

begin
    KeyboardState := STATE_SETFREQ;
    LBWARN.Visible := TRUE;
    RXState.Tune := FALSE;
    LBFAULT.Hide;
    SetActiveButtons;
    PutRXString('-0000.000');
    MsgToShow := ENTERFREQKHZ;
end;

// "CHAN"
procedure TRA3701.BCHANClick(Sender: TObject);

var s,schn: string;

begin
  MsgToShow := ENTERCHANNO;
  KeyboardState := STATE_SELCHAN;
  OldRXState := RXState;
  if MUSEPROGCH.Checked then begin
    iChNum := LastChan;
     s := Format('%0.2D',[ichNum]);
    PutChanString(s);
    RXState := RXChannels[iChnum];
    UpdateLeftPanel;
    DoMessages;
  end else begin
    if In_Command then exit;
    In_Command := TRUE;
    // Get actual channel number
    if SendCommand(Get_Chan,'') then begin
        if GetReply then begin
          TryStrToInt(Copy(Status,5,2),iChNum);
          LastChan := iChNum;
          rCurChan := iChNum;
          CurChan := iChNum;
          s := Format('%0.2D',[ichNum]);
          PutChanString(s);
          UpdateLeftPanel;
          DoMessages;
        end else MsgToShow := NOREPLY;
    end else MsgToShow := COMMANDFAILED;

    // Enter delta mode
    if SendCommand(Set_DeltaMode, '') then begin
      DeltaMode_ON := TRUE;
      if GetReply then begin
        FmtStr(schn,'%02D',[iChNum]);
        if SendCommand(Set_Chan,schn) then begin
            if GetReply then begin
               GetStatus;
               if not ParseStatus then MsgToShow := PARSESTFAILED;
            end else MsgToShow := NOREPLY;
        end else MsgToShow := COMMANDFAILED;
      end else MsgToShow := NOREPLY;
    end else MsgToShow := COMMANDFAILED;
  end;
  In_Command := FALSE;
end;

// "SCAN"
procedure TRA3701.BSCANClick(Sender: TObject);

var StartCh, StopCh, DwellT, CORC, CORL, Dir, par: string;
    p: integer;

begin
  if MUSEPROGCH.Checked then begin   // use program channels
    case KeyboardState of
      STATE_NORMAL: begin
        // Show parameters window and go in the SCAN state
        KeyboardState := STATE_SCAN;
        FCHSCAN.Show;
        FCHSCAN.SPCHSTART.SetFocus;
      end;
      STATE_SELCHAN, STATE_STOCHAN: begin
        // pressing CHAN in this state sets/clears the channel SCAN flag
        ichnum := LastChan;
        RXChannels[ichnum].ScanFlag := not RXChannels[ichnum].ScanFlag;
        LBSCAN.Visible := RXChannels[ichnum].ScanFlag;
      end;
    end;
  end else begin // Use receiver channels
    case KeyboardState of
      STATE_NORMAL: begin
        SetActiveButtons;
        OldRXState := RXState;
        // Retrieve current parameters (reply is something like SCCH00,99,01.50,1;)
        if In_Command then exit;
        In_Command := TRUE;
        if SendCommand(Get_ScanParms, '') then begin
          if GetReply then begin
            p := Pos('SCCH',Status);
            if p > 0 then begin
              StartCh := Copy(Status,p+4,2);
              StopCh  := Copy(Status,p+7,2);
              DwellT  := Copy(Status,p+10,5);
              CORC    := Copy(Status,p+16,1);
              fchscan.SPCHSTART.Value := StrToInt(StartCh);
              fchscan.SPCHEND.Value :=  StrToInt(StopCh);
              fchscan.FSPDWELL.Value := StrToFloat(AdjustDecimalSeparator(DwellT));
              fchscan.CBCOR.ItemIndex := StrToInt(CORC);
            end;
          end else MsgToShow := NOREPLY;
        end else MsgToShow := COMMANDFAILED;

        // Retrieve scan direction
        if SendCommand(Get_Scan, '') then begin
          if GetReply then begin
            p := Pos('SCAN',Status);
            if p > 0 then begin
              Dir := Copy(Status,p+6,1);
              fchscan.CBDIR.ItemIndex := StrToInt(Dir);
            end;
          end else MsgToShow := NOREPLY;
        end else MsgToShow := COMMANDFAILED;

        // Retrieve COR level
        if SendCommand(Get_CORSQLevel, '') then begin
          if GetReply then begin
            p := Pos('CORL',Status);
            if p > 0 then begin
              CORL := Copy(Status,p+4,3);
              fchscan.SPCORLEVEL.Value := StrToInt(CORL);
            end;
          end else MsgToShow := NOREPLY;
        end else MsgToShow := COMMANDFAILED;
        In_Command := FALSE;

        // Show parameters window and go in the SCAN state
        FCHSCAN.Show;
        FCHSCAN.SPCHSTART.SetFocus;
        KeyboardState := STATE_SCAN;
      end;
      STATE_SELCHAN: begin
        // pressing CHAN in this state sets/clears the channel SCAN flag
        LBSCAN.Visible := not LBSCAN.Visible;
        if LBSCAN.Visible then par := '1' else par := '0';
        if not SendCommand(Set_ScanFlag, par) then begin
          MsgToShow := COMMANDFAILED;
        end;
      end;
    end;
  end;
end;

// "Store"
procedure TRA3701.BSTOREClick(Sender: TObject);

var s,schn: string;

begin
  MsgToShow := ENTERCHANNO;
  KeyboardState := STATE_STOCHAN;
  OldRXState := RXState;
  if MUSEPROGCH.Checked then begin
    iChNum := LastChan;
     s := Format('%0.2D',[ichNum]);
    PutChanString(s);
    RXState := RXChannels[iChnum];
    UpdateLeftPanel;
    DoMessages;
  end else begin
    if In_Command then exit;
    In_Command := TRUE;
    // Get actual channel number
    if SendCommand(Get_Chan,'') then begin
      if GetReply then begin
        TryStrToInt(Copy(Status,5,2),iChNum);
        LastChan := iChNum;
        rCurChan := iChNum;
        CurChan := iChNum;
        s := Format('%0.2D',[ichNum]);
        PutChanString(s);
        UpdateLeftPanel;
        DoMessages;
      end else MsgToShow := NOREPLY;
    end else MsgToShow := COMMANDFAILED;

    // Enter delta mode
    if SendCommand(Set_DeltaMode, '') then begin
        DeltaMode_ON := TRUE;
        if GetReply then begin
          FmtStr(schn,'%02D',[iChNum]);
          if SendCommand(Set_Chan,schn) then begin
            if GetReply then begin
              GetStatus;
              if not ParseStatus then MsgToShow := PARSESTFAILED;
            end else MsgToShow := NOREPLY;
          end else MsgToShow := COMMANDFAILED;
        end else MsgToShow := NOREPLY;
    end else MsgToShow := COMMANDFAILED;
  end;
  In_Command := FALSE;
end;

// "1"
procedure TRA3701.B1Click(Sender: TObject);

begin
    if In_Command then exit;
    In_Command := TRUE;
    KeyboardHandler('1');
    In_Command := FALSE;
end;

// "2"
procedure TRA3701.B2Click(Sender: TObject);

begin
    if In_Command then exit;
    In_Command := TRUE;
    KeyboardHandler('2');
    In_Command := FALSE;
end;

// "3/BW3"
procedure TRA3701.B3Click(Sender: TObject);

begin
    if In_Command then exit;
    In_Command := TRUE;
    KeyboardHandler('3');
    In_Command := FALSE;
end;

// "4"
procedure TRA3701.B4Click(Sender: TObject);

begin
    if In_Command then exit;
    In_Command := TRUE;
    KeyboardHandler('4');
    In_Command := FALSE;
end;

// "5"
procedure TRA3701.B5Click(Sender: TObject);

begin
    if In_Command then exit;
    In_Command := TRUE;
    KeyboardHandler('5');
    In_Command := FALSE;
end;

// "6"
procedure TRA3701.B6Click(Sender: TObject);

begin
    if In_Command then exit;
    In_Command := TRUE;
    KeyboardHandler('6');
    In_Command := FALSE;
end;

// "7"
procedure TRA3701.B7Click(Sender: TObject);

begin
    if In_Command then exit;
    In_Command := TRUE;
    KeyboardHandler('7');
    In_Command := FALSE;
end;

// "8"
procedure TRA3701.B8Click(Sender: TObject);

begin
    if In_Command then exit;
    In_Command := TRUE;
    KeyboardHandler('8');
    In_Command := FALSE;
end;

// "9"
procedure TRA3701.B9Click(Sender: TObject);

begin
    if In_Command then exit;
    In_Command := TRUE;
    KeyboardHandler('9');
    In_Command := FALSE;
end;

// "0"
procedure TRA3701.B0Click(Sender: TObject);

begin
    if In_Command then exit;
    In_Command := TRUE;
    KeyboardHandler('0');
    In_Command := FALSE;
end;

// "RCL"
procedure TRA3701.BRCLClick(Sender: TObject);
begin
  case KeyboardState of
    STATE_MENU..STATE_MENU_M4: begin
      Keyboardstate := STATE_NORMAL;
      MenuLevel := 1;
      UpdateDisplay;
      exit;
    end;
    STATE_BITESTOP: begin
      // BITE stopped for an error, resume from next test
      if In_Command then exit;
      In_Command := TRUE;
      if SendCommand(Set_BITE,slevel+','+snexttest) then begin
        if not GetReply then begin
          MsgToShow := NOREPLY;
        end;
      end else Message (COMMANDFAILED);
      RCLPressed := TRUE;
    end;
    STATE_BITEFACT,STATE_BITECONF,STATE_BITEEND,STATE_BITEFF: begin
      // BITE has finished
      In_Command :=  TRUE;
      if SendCommand(Recall,'') then begin
        if GetReply then begin
          KeyboardState := STATE_NORMAL;
        end else MsgToShow := NOREPLY;
      end else Message (COMMANDFAILED);
      // Restore previous RX state, Smeter and continuous BITE timers status
      RXstate := OldRXState;
      TSMeter.Enabled := OldTSen;
      TBITE.Enabled := OldTBEn;
      KeyboardState := STATE_NORMAL;
      SetActiveButtons;
      LBBITE.Caption := '';
      RCLPressed := TRUE;
      In_Command := FALSE;
    end;
    STATE_NORMAL: begin
    // RCL in normal RX state does nothing
    end;
    STATE_SELCHAN, STATE_STOCHAN: begin
      // Exit delta mode
      if SendCommand(Recall, '') then begin
        if not GetReply then MsgToShow := NOREPLY;
      end else MsgToShow := COMMANDFAILED;
      KeyboardState := STATE_NORMAL;
      DeltaMode_ON := FALSE;
      RXstate := OldRXState;
      UpdateDisplay;
    end;
    STATE_SETADDR: begin
      UpdateDisplay;
      KeyboardState := STATE_NORMAL;
    end;
  end;
  In_Command := FALSE;
end;

// "TUNE +"
procedure TRA3701.BTUNEPClick(Sender: TObject);
begin
  RXState.Tune := TRUE;
  case RXState.Freq_step of
    -1: begin
      RXState.Freq_step := 1;
      MsgToShow := STEP1HZSEL;
    end;
    1: begin
      RXState.Freq_step := 10;
      MsgToShow := STEP10HZSEL;
    end;
    10: begin
      RXState.Freq_step := 100;
      MsgToShow := STEP100HZSEL;
    end;
    100: begin
      RXState.Freq_step := 1000;
      MsgToShow := STEP1000HZSEL;
    end;
    1000: BEGIN
      RXState.Freq_step := 1000;
      MsgToShow := STEP1000HZSEL;
    end;
    otherwise begin
      RXState.Freq_step := 1;
      MsgToShow := STEP1HZSEL;
    end;
  end;
  LBKHZT.Show;
  UpdateDisplay;
  UpdateSpinAndCombos;
end;

// "TUNE-"
procedure TRA3701.BTUNEMClick(Sender: TObject);
begin

//  RXstate.Tune := TRUE;
  case RXState.Freq_step of
    -1,1: begin
      RXState.Tune := FALSE;
      RXState.Freq_Step := -1;
      MsgToShow := TUNINGDISABLED;
    end;
    10: begin
      RXState.Freq_step := 1;
      MsgToShow := STEP1HZSEL;
    end;
    100: begin
      RXState.Freq_step := 10;
      MsgToShow := STEP10HZSEL;
    end;
    1000: begin
      RXState.Freq_step := 100;
      MsgToShow := STEP100HZSEL;
    end;
    otherwise begin
      MsgToShow := STEP1HZSEL;
      RXState.Freq_step := 1;
    end;
  end;
  if RXstate.Tune then LBKHZT.Show else LBKHZT.Hide;
  UpdateDisplay;
  UpdateSpinAndCombos;
end;

// BFO
procedure TRA3701.BBFOClick(Sender: TObject);
begin
  if KeyboardState = STATE_NORMAL then begin
    KeyboardState := STATE_SETBFO;
    RXState.Tune := FALSE;
    LBBFO.Show;
    LBLOCK.Hide;
    MsgToShow := SETBFOFREQ;
    SetActiveButtons
  end;
end;

// "ENTER"
procedure TRA3701.BENTERClick(Sender: TObject);

var p: integer;
    s: string;
    tmpRXAddress: integer;

begin
  In_Command := TRUE;
  case KeyboardState of
    STATE_SETFREQ: begin
      s := GetRXString;
      p := pos('-',s);
      if p <> 0 then s[p] := '0';
      RXState.Freq_rx := Trunc(1000*StrToFloat(AdjustDecimalSeparator(s)));
      SetRXfreq;
      KeyboardState := STATE_NORMAL;
     end;
     STATE_SELCHAN: begin
         ichnum := LastChan;
       if MUSEPROGCH.Checked then begin
         RXState := RXChannels[ichnum];
         RXState.Tune := FALSE;
         UpdateDisplay;
         RestoreState;
         LastChan := ichnum;
         CurChan := ichnum;
         KeyboardState := STATE_NORMAL;
       end else begin
         // Exit delta mode
         if SendCommand(Recall, '') then begin
           if not GetReply then MsgToShow := NOREPLY;
         end else MsgToShow := COMMANDFAILED;

        if SendCommand(Set_Chan,s) then begin
           if GetReply then begin
             GetStatus;
             if not ParseStatus then MsgToShow := PARSESTFAILED;
             UpdateDisplay;
             KeyboardState := STATE_NORMAL;
           end else MsgToShow := NOREPLY;
         end else MsgToShow := COMMANDFAILED;
       end;
     end;
     STATE_STOCHAN: begin
       ichnum := LastChan;
       if MUSEPROGCH.Checked then begin
         RXChannels[ichnum] := OldRXState;
         // update scan flag
         RXchannels[ichnum].ScanFlag := LBSCAN.Visible;
         RXState := OldRXState;
         RXState.Tune := FALSE;
         UpdateDisplay;
         LastChan := ichnum;
         CurChan := ichnum;
         KeyboardState := STATE_NORMAL;
       end else begin
         // Exit delta mode
         if SendCommand(Recall, '') then begin
           if not GetReply then MsgToShow := NOREPLY;
         end else MsgToShow := COMMANDFAILED;

         if SendCommand(Store_Channel,s) then begin
           if GetReply then begin
             GetStatus;
             if not ParseStatus then MsgToShow := PARSESTFAILED;
             UpdateDisplay;
             KeyboardState := STATE_NORMAL;
           end else MsgToShow := NOREPLY;
         end else MsgToShow := COMMANDFAILED;
       end;
     end;
     STATE_SETBFO: begin
       KeyboardState := STATE_NORMAL;
     end;
     STATE_SETADDR: begin
       if TryStrToInt(GetAddrString, tmpRXAddress) then begin
         // Close comms with the previous RX
         KeyboardState := STATE_NORMAL;
         In_Command := FALSE;
         SHSWMouseUp(Sender,mbLeft,[],0,0);

         // Set new address in current RX#
         CommParms.RXAddresses[CurRXNum] := tmpRXAddress;
         CurRXAddress := tmpRXAddress;
         SaveConfig;

         // Open comms with the new RX
         SHSWMouseUp(Sender,mbLeft,[],0,0);
       end;
     end;
     otherwise begin
       KeyboardState := STATE_NORMAL;
       UpdateRightPanel;
     end;
  end;
  UpdateDisplay;
  SetActiveButtons;
  In_Command := FALSE;
end;

//
// Simulated tune knob
//
// Mouse move on tuning knob
procedure TRA3701.SHKNOB1MouseMove(Sender: TObject; Shift: TShiftState; X,
    Y: Integer);

var alpha, dalpha, myx, myy, deltatms: real;
    tmpstep,step,direction: integer;
    s: string;
    t: TTime;

begin
  if In_KnobMouseMove or In_Command then exit;
  if not (KeyboardState in [STATE_NORMAL..STATE_STOCHAN]) then exit;
  if (KeyboardState = STATE_NORMAL) and not RXState.Tune then exit;

  In_Command := TRUE;
  In_KnobMouseMove := TRUE;

  if MTIMED.Checked then begin
    t := Time;
    deltatms := (t-LastTuningTime)/OneMillisecond;
    LastTuningTime := t;
    if deltatms>1000 then deltatms := 1000;
    {$IFDEF TEST_KNOBMOUSEMOVE}
    MsgToShow := 'deltat = '+IntToStr(trunc(deltatms))+ ' ms';
    {$ENDIF}
  end;

  if (sqr(X-X2)+sqr(Y-Y2)) > sqr(R2) then begin
    In_KnobMouseMove := FALSE;
    In_Command := FALSE;
    exit;
  end;

  Y := Y+KnobOffsetY;
  X := X+KnobOffsetX;
  alpha := arctan2(Y-SHKNOB1.Height / 2, X - SHKNOB1.Width / 2);
  myx := r * cos(alpha) + cx;
  myy := r * sin(alpha) + cy;
  SHKNOB.Top := trunc(myy);
  SHKNOB.Left := trunc(myx);
  {$IFDEF TEST_KNOBMOUSEMOVE}
  MsgToShow := 'alpha='+floatTostr(alpha)+' prevalpha='+floatTostr(prevalpha);
  {$ENDIF}
  alpha := alpha+pi; // make angle interval positive (0..2*pi)
  dalpha := alpha-prevalpha;
  if (dalpha) < -pi then begin
      prevalpha := 0;   // (2*pi) --> 0
      dalpha := alpha-prevalpha;
  end;
  if (dalpha) > pi then begin
      prevalpha := 2*pi; // 0 --> 2*pi
      dalpha := alpha-prevalpha;
  end;
  if dalpha > 0 then direction := 1 else direction := -1;
  tmpstep := trunc(HzPerTurn*((dalpha)/(2*pi)));
  prevalpha := alpha;

  if MTIMED.Checked then begin
    case trunc(deltatms) of
      0..25:   step := 10000;
      26..200: step := 1000;
      201..500: step := 100;
      501..800: step := 10;
      801..1000: step := 1;
    end;
    step := step * direction;
    {$IFDEF TEST_KNOBMOUSEMOVE}
    MsgToShow := 'step = '+IntToStr(step)+ ' Hz';
    {$ENDIF}
  end else begin
    step := tmpstep;
  end;
  DoMessages;
  case KeyboardState of
    STATE_NORMAL: begin
      RXState.Freq_RX := RXState.Freq_rx+step;
      if RXState.Freq_RX > MAXRXF then begin
          RXState.Freq_RX := MAXRXF;
          MsgToShow := RXCANTTUNE;
      end;
      if RXState.Freq_RX < MINRXF then begin
          RXState.Freq_RX := MINRXF;
          MsgToShow := RXCANTTUNE;
      end;
      SetRXFreq;
      DispRXF;
    end;
    STATE_SETBFO: begin
      if step > 0 then step := 10 else step := -10;
      RXstate.BFO := RXstate.BFO + step;
      if RXstate.BFO > 9900 then RXState.BFO := 9900;
      if RXstate.BFO < -9900 then RXState.BFO := -9900;
      In_Command := TRUE;
      if SendCommand(Set_BFO, IntToStr(RXState.BFO)) then begin
        if not GetAck then MsgToShow := COMMANDFAILED
      end else MsgToShow := COMMANDFAILED;
      In_Command := FALSE;
      UpdateDisplay;
    end;
    STATE_SELCHAN, STATE_STOCHAN: begin
      if step > 0 then
        rCurChan := rCurChan + 0.2
      else
        rCurChan := rCurChan - 0.2;
      if rCurChan < 0 then rCurChan := 99;
      if rCurChan > 99 then rCurChan := 0;
      CurChan := trunc(rCurChan);
      if CurChan <> OldCurChan then begin
        OldCurchan := CurChan;
        s := Format('%2.2D',[CurChan]);
        PutChanString(s);
        LastChan := CurChan;
        if MUSEPROGCH.Checked then begin
          if KeyboardState = STATE_SELCHAN then begin
            RXState := RXChannels[CurChan];
            UpdateDisplay;
          end;
        end else begin
          // We are still in delta mode, retrieve channel info and display them
          // The receiver continues to receive at the previous frequency
          if SendCommand(Set_Chan,s) then begin
            if GetReply then begin
              if SendCommand(Get_All, '') then begin
                if GetReply then begin
                  if not ParseStatus then MsgToShow := PARSESTFAILED;
                 end else MsgToShow := NOREPLY;
              end else MsgToShow := COMMANDFAILED;
            end else MsgToShow := NOREPLY;
          end else MsgToShow := COMMANDFAILED;
        end;
        UpdateLeftPanel;
        UpdateRightPanel;
        DoMessages;
      end;
    end;
  end;
  In_KnobMouseMove := FALSE;
  In_Command := FALSE;
end;

// Click on the tuning knob
procedure TRA3701.SHKNOBMouseUp(Sender: TObject; Button: TMouseButton;
    Shift: TShiftState; X, Y: Integer);
begin
  case KeyboardState of
    STATE_NORMAL: begin
      RXState.Tune := not RXState.Tune;
      if RXstate.Tune then begin
        SPFREQ.Enabled := TRUE;
        LastTuningTime := Time;
        LBKHZT.Caption := IntToStr(HzPerTurn div 1000)+KHZT;
      end else begin
        SPFREQ.Enabled := FALSE;
        LBKHZT.Hide;
      end;
      UpdateDisplay;
    end;
    STATE_SELCHAN,STATE_STOCHAN,STATE_SETBFO: BEnterCLick(Sender);
  end;
end;


//
// Rightmost buttons block
//

// "M1"
procedure TRA3701.BM1Click(Sender: TObject);

var subLevel: integer;

begin
  {$IFDEF RA3701MENUS}
  if KeyboardState <> STATE_MENU then exit;
  KeyboardState := STATE_MENU_M1;
  case MenuLevel of
    1: begin
     LBDISPLINE1.Caption := MenuLevel1_1[1];
     LBDISPLINE2.Caption := '';
   end;
    2: begin
    end;
  end;
  {$ENDIF}
end;

// "M2"
procedure TRA3701.BM2Click(Sender: TObject);
begin
  {$IFDEF RA3701MENUS}
  if KeyboardState <> STATE_MENU then exit;
  KeyboardState := STATE_MENU_M2;
  case MenuLevel of
    1: begin
     LBDISPLINE1.Caption := MenuLevel1_2[1];
     LBDISPLINE2.Caption := '';
    end;
    2: begin
    end;
  end;
  {$ENDIF}
end;

// "M3"
procedure TRA3701.BM3Click(Sender: TObject);
begin
  {$IFDEF RA3701MENUS}
  if KeyboardState <> STATE_MENU then exit;
  KeyboardState := STATE_MENU_M3;
  case MenuLevel of
    1: begin
      LBDISPLINE1.Caption := MenuLevel1_3;
      LBDISPLINE2.Caption := '';
    end;
    2: begin
    end;
  end;
  {$ENDIF}
end;

// "M4"
procedure TRA3701.BM4Click(Sender: TObject);

var st1,st2: string;
  p,l: integer;

begin
  {$IFDEF RA3701MENUS}
  if KeyboardState <> STATE_MENU then exit;
  KeyboardState := STATE_MENU_M4;
   case MenuLevel of
    1: begin
      p := Pos('|', MenuLevel1_4[1]);
      l := Length(MenuLevel1_4[1]);
      st1 := Copy(MenuLevel1_4[1], 1, p-1);
      st2 := Copy(MenuLevel1_4[1], p+1, l-p);
      LBDISPLINE1.Caption := Format(st1, [Float(RXState.Bandwidth div 1000)]);
      LBDISPLINE2.Caption := st2;
    end;
    2: begin
    end;
  end;
  {$ENDIF}
end;

// "METER"
procedure TRA3701.BMETERClick(Sender: TObject);
begin
  // No remote command to set/read RX meter mode so modify only the program display
  RXState.MeterMode := not RXstate.MeterMode;
  UpdateCenterPanel;
end;

// "MENU"
procedure TRA3701.BMENUClick(Sender: TObject);
begin
  if KeyboardState < STATE_MENU then begin
    MenuLevel := 1;
    KeyboardState := STATE_MENU;
    UpdateRightPanel;
  end else begin
    Inc(MenuLevel);
    if MenuLevel > 6 then MenuLevel := 1;
  end;
  DisplayTopMenu(MenuLevel);
end;

// RUN/HOLD (No remote command)
procedure TRA3701.BRUNHClick(Sender: TObject);
begin
  MsgToShow := NOREMRH;
end;

// "L/S"
procedure TRA3701.BLSClick(Sender: TObject);

var tmpmute: boolean;
    par: string;

begin
  if In_Command then exit;
  In_Command := TRUE;
  tmpmute := not RXstate.Mute;
  if tmpmute then
    par := '1'
  else
    par := '0';
  if SendCommand(Set_Mute, par) then begin
    RXState.Mute := tmpmute;
    if tmpmute then MsgToShow := WAUDIOMUTED;
  end else MsgToShow := COMMANDFAILED;
  In_Command := FALSE;
  UpdateCenterPanel;
end;

// "AUX"
procedure TRA3701.BAUXClick(Sender: TObject);

var AUXFile: file of TRXState;
    tmpRXState: TRXstate;

begin
  // Mamual says AUX mode is M0, but my receiver complains PARAMETER OUT OF RANGE
  // so this command is used to load the AUXConf.dat file, if present.

{$IFDEF RXAUXMODE}
if In_Command then exit;
In_Command := TRUE;
if SendCommand(Set_Mode, IntToStr(M_AUX)) then begin
    IF GetReply then begin
      if ParseStatus then begin
        RXstate.Mode_RX := M_AUX;
        In_Command := FALSE;
        GetStatus;
        if not ParseStatus then MsgToShow := PARSESTFAILED;
        SetBWCombo;
        SetActiveButtons;
        UpdateDisplay;
        UpdateSpinAndCombos;
      end;
    end else MsgToShow := NOREPLY;
  end;
end;
In_Command := FALSE;
{$ELSE}
  AssignFile(AUXFile, Homedir+AUXFileName);
  try
    Reset(AUXFile);
    Read(AUXFile,tmpRXState);
    CloseFile(AUXFile);
  except
    MsgToShow := NOAUXFILE;
    In_Command := FALSE;
    exit;
  end;
   RXState := tmpRXState;
   RestoreState;
   UpdateDisplay;
   UpdateSpinAndCombos;
   MsgToShow := AUXFILELOADED;
{$ENDIF}
end;

// "FSK"
procedure TRA3701.BFSKClick(Sender: TObject);
begin
  if In_Command then exit;
  In_Command := TRUE;
  if SendCommand(Set_Mode, IntToStr(M_FSK)) then begin
    if GetReply then begin
      if ParseStatus then begin
        GetStatus;
        SetActiveButtons;
        UpdateDisplay;
      end;
    end else MsgToShow := NOREPLY;
  end else MsgToShow := COMMANDFAILED;
  In_Command := FALSE;
end;

// "ISB"
procedure TRA3701.BISBClick(Sender: TObject);

var tmpmode: integer;

begin
  if In_Command then exit;
  In_Command := TRUE;
  if (RXState.Mode_RX <> M_ISBU) then
    tmpmode := M_ISBU
  else
    tmpmode := M_ISBL;
  if SendCommand(Set_Mode, IntToStr(tmpmode)) then begin
    if GetReply then begin
      if ParseStatus then begin
        GetStatus;
        SetActiveButtons;
        UpdateRightPanel;
        UpdateSpinAndCombos;;
        SetBWCombo;
      end;
    end else MsgToShow := NOREPLY;
  end else MsgToShow := COMMANDFAILED;
  In_Command := FALSE;
end;

// "LSB"
procedure TRA3701.BLSBClick(Sender: TObject);

begin
  if In_Command then exit;
  In_Command := TRUE;
  if SendCommand(Set_Mode, IntToStr(M_LSB)) then begin
    if GetReply then begin
      GetStatus;
      SetActiveButtons;
      UpdateRightPanel;
      UpdateSpinAndCOmbos;
      SetBWCombo;
    end else MsgToShow := NOREPLY;
  end else MsgToShow := COMMANDFAILED;
  In_Command := FALSE;
end;

// "USB"
procedure TRA3701.BUSBClick(Sender: TObject);
begin
  if In_Command then exit;
  In_Command := TRUE;
  if SendCommand(Set_Mode, IntToStr(M_USB)) then begin
    if GetReply then begin
      GetStatus;
      SetActiveButtons;
      UpdateRightPanel;
      UpdateSpinAndCombos;
      SetBWCombo;
    end else MsgToSHow := NOREPLY;
  end else MsgToShow := COMMANDFAILED;
  In_Command := FALSE;
end;

// "AM"
procedure TRA3701.BAMClick(Sender: TObject);
begin
  if In_Command then exit;
  In_Command := TRUE;
  if SendCommand(Set_Mode, IntToStr(M_AM)) then begin
    if GetReply then begin
      GetStatus;
      SetActiveButtons;
      UpdateRightPanel;
      UpdateSpinAndCombos;
      SetBWCombo;
    end else MsgToShow := NOREPLY;
  end else MsgToShow := COMMANDFAILED;
  In_Command := FALSE;
end;

// "CW"
procedure TRA3701.BCWClick(Sender: TObject);
begin
  if In_Command then exit;
  In_Command := TRUE;
  if SendCommand(Set_Mode, IntToStr(M_CW)) then begin
    if GetReply then begin
      GetStatus;
      SetActiveButtons;
      UpdateRightPanel;
      UpdateSpinAndCombos;
      SetBWCombo;
    end else MsgToShow := NOREPLY;
  end else MsgToShow := COMMANDFAILED;
  In_Command := FALSE;
end;

// "FM"
procedure TRA3701.BFMClick(Sender: TObject);
begin
  if In_Command then exit;
  In_Command := TRUE;
  if SendCommand(Set_Mode, IntToStr(M_FM)) then begin
    if GetReply then begin
      GetStatus;
      SetActiveButtons;
      UpdateRightPanel;
      UpdateSpinAndCombos;
      SetBWCombo;
    end else MsgToShow := NOREPLY;
  end else MsgToShow := COMMANDFAILED;
  In_Command := FALSE;
end;

// "SQLCH"
procedure TRA3701.BSQLClick(Sender: TObject);

var par: string;
    tmpState: boolean;

begin
    if In_Command then exit;
    In_Command := TRUE;
    tmpState := not RXState.SquelchEnabled;
    if tmpstate then par := '1' else par := '0';

    // Squelch does not work in MAN modes
    if tmpState and (RXState.AGCMode in [MANONLY,MANT]) then begin
      MsgToShow := DONTWORKINMAN;
    end;

    if SendCommand(Set_Squelch, par) then begin
      if GetReply then begin
        if SendCommand(Get_Squelch, '') then begin
          if GetReply then begin
            if ParseStatus then begin
              UpdateRightPanel;
              UpdateIFGainAndSquelch;
              UpdateSquelchLabel;
            end else MsgToShow := PARSESTFAILED;
          end else MsgToShow := NOREPLY;
        end else MsgToShow := COMMANDFAILED;
      end else MsgToShow := NOREPLY;
    end else MsgToShow := COMMANDFAILED;
    In_Command := FALSE;
end;

// "MAN"
procedure TRA3701.BMANClick(Sender: TObject);

var NewAGCMode: integer;

begin
  if In_Command then exit;
  In_Command := TRUE;
  NewAGCMode := RXState.AGCMode+1;
  if NewAGCMode > MANT then NewAGCMode := AGCON;
  if SendCommand(Set_AGC, IntToStr(NewAGCMode)+','+IntTostr(RXState.AGCTimeCon)) then begin
    if GetReply then begin
      if SendCommand(Get_AGC, '') then begin
        if GetReply then begin
          if ParseStatus then begin
            SetActiveButtons;
            UpdateRightPanel;
            UpdateSpinAndCombos;
            UpdateIFGainAndSquelch;
          end else MsgToShow := PARSESTFAILED;
        end else MsgToShow := NOREPLY;
      end else MsgToShow := COMMANDFAILED;
    end else MsgToShow := NOREPLY;
  end else MsgToShow := COMMANDFAILED;

  // Check squelch
  if RXstate.SquelchEnabled and (RXState.AGCMode in [MANONLY,MANT]) then begin
    MsgToShow := DONTWORKINMAN;
    GetStatus;
    if ParseStatus then UpdateRightPanel;
  end;
  In_Command := FALSE;
end;

// "AGC -"
procedure TRA3701.BAGCMClick(Sender: TObject);

var NewAGCTimeCon: integer;

begin
  if In_Command then exit;
  In_Command := TRUE;
  NewAGCTimeCon := RXState.AGCTimeCon-1;
  if NewAGCTimeCon < 0 then NewAGCTimeCon := 0;
  if SendCommand(Set_AGC, IntToStr(RXState.AGCMode)+','+IntTostr(NewAGCTimeCon)) then begin
    if GetReply then begin
      if SendCommand(Get_AGC, '') then begin
        if GetReply then begin
          if ParseStatus then begin
            RXstate.AGCTimeCon := NewAGCTimeCon;
            SetActiveButtons;
            UpdateDisplay;
            UpdateSpinAndCombos;
          end else MsgToSHow := PARSESTFAILED;
        end else MsgToShow := NOREPLY;
      end else MsgToShow := NOREPLY;
    end else MsgToShow := COMMANDFAILED;
  end else MsgToShow := COMMANDFAILED;
  In_Command := FALSE;
end;

// "AGC +"
procedure TRA3701.BAGCPClick(Sender: TObject);

var NewAGCTimeCon: integer;

begin
  if In_Command then exit;
  In_Command := TRUE;
  NewAGCTimeCon := RXstate.AGCTimeCon+1;
  if NewAGCTimeCon > 4 then NewAGCTimeCon := 4;
  if SendCommand(Set_AGC, IntToStr(RXState.AGCMode)+','+IntTostr(NewAGCTimeCon)) then begin
    if GetReply then begin
      if SendCommand(Get_AGC, '') then begin
        if GetReply then begin
          if ParseStatus then begin
            RXstate.AGCTimeCon := NewAGCTimeCon;
            SetActiveButtons;
            UpdateDisplay;
            UpdateSpinAndCombos;
          end else MsgToShow := PARSESTFAILED;
        end else MsgToShow := NOREPLY;
      end else MsgToShow := COMMANDFAILED;
    end else MsgToShow := NOREPLY;
  end else MsgToShow := COMMANDFAILED;
  In_Command := FALSE;
end;

// "BW -"
procedure TRA3701.BBWMClick(Sender: TObject);

var stmp: string;
    BW: array[0..15] of TBandwidth;
    maxBW: integer;

begin
  if In_Command then exit;
  In_Command := TRUE;
  case RXstate.Mode_RX of
    M_USB: begin
      maxBW := NumUSBBW-1;
      BW := USBBandwidths;
    end;
    M_LSB: begin
      maxBW := NumLSBBW-1;
      BW := LSBBandwidths;
    end;
    otherwise begin
      maxBW := NumSYMBW-1;
      BW := SYMBandwidths;
    end;
  end;
  BWNum := BWNum-1;
  if BWNum>maxBW then BWNum := maxBW;
  if BWNum<0 then BWNum := 0;
  FmtStr(stmp,'%1D,%0.2D',[BW[BWNum].BWType, BW[BWNum].BWNumber]);
  if SendCommand(Set_IFBW, stmp) then begin
    if GetReply then begin
      if SendCommand(Get_IFBW,'') then begin
        if GetReply then begin
          if ParseStatus then begin
            RXState.Bandwidth := BW[BWNum].FHigh-BW[BWNum].FLow;
            In_Command := FALSE;
            ParseStatus;
            UpdateRightPanel;
            UpdateSpinAndCombos;
            SetBWCombo;
          end else MsgToShow := PARSESTFAILED;
        end else MsgToShow := NOREPLY;
      end else MsgToShow := COMMANDFAILED;
    end else MsgToShow := NOREPLY;
  end else MsgToShow := COMMANDFAILED;
  In_Command := FALSE;
  DoMessages;
end;

// "BW +"
procedure TRA3701.BBWPClick(Sender: TObject);

var stmp: string;
  maxBW: integer;
  BW: array[0..15] of TBandwidth;

begin
  if In_Command then exit;
  In_Command := TRUE;
  case RXstate.Mode_RX of
    M_USB: begin
      maxBW := NumUSBBW-1;
      BW := USBBandwidths;
    end;
    M_LSB: begin
      maxBW := NumLSBBW-1;
      BW := LSBBandwidths;
    end;
    otherwise begin
      maxBW := NumSYMBW-1;
      BW := SYMBandwidths;
    end;
  end;
  BWNum := BWNum+1;
  if BWNum > maxBW then BWNum := maxBW;
  FmtStr(stmp,'%1D,%0.2D',[BW[BWNum].BWType, BW[BWNum].BWNumber]);
  if SendCommand(Set_IFBW,stmp) then begin
    if GetReply then begin
      if SendCommand(Get_IFBW,'') then begin
        if GetReply then begin
          if ParseStatus then begin
            RXState.Bandwidth := BW[BWNum].FHigh-BW[BWNum].FLow;
            In_Command := FALSE;
            UpdateRightPanel;
            UpdateSpinAndCombos;
            SetBWCombo;
          end else MsgToShow := PARSESTFAILED;
        end else MsgToShow := NOREPLY;
      end else MsgToShow := COMMANDFAILED;
    end else MsgToShow := NOREPLY;
  end else MsgToShow := COMMANDFAILED;
  In_Command := FALSE;
end;

//
// IF Gain and volume knobs
//
// Mouse entering simulated IF Gain knob
procedure TRA3701.SHRFGMouseEnter(Sender: TObject);
begin
  if In_ChanUpdate then exit;
  if KeyboardState <> STATE_POWEROFF then begin
    MsgToShow := USEMOUSEWHEEL;
    if (RXstate.AGCMode = AGCON) and not RXState.SquelchEnabled then
      MsgToShow := NOMANIFGSQ;
  end;
end;

// Mouse wheel on the simulated IF Gain knob
procedure TRA3701.SHRFGMouseWheel(Sender: TObject; Shift: TShiftState;
  WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);

var setcmd, getcmd, par: string;

begin
  if In_ChanUpdate then exit;
  if KeyboardState = STATE_POWEROFF then exit;

  if In_Command then begin
    Handled := TRUE;
    exit;
  end;

  In_Command := TRUE;

  if WheelDelta > 0 then begin
    if RXstate.SquelchEnabled then begin
      Inc(RXState.CORSQLevel);
      if RXState.CORSQLevel > 255 then RXState.CORSQLevel := 255;
    end else begin
      Inc(RXState.IFGain);
      if RXState.IFGain > 255 then RXState.IFGain := 255;
    end;
  end;
  if WheelDelta < 0 then begin
    if RXstate.SquelchEnabled then begin
      Dec(RXState.CORSQLevel);
      if RXState.CORSQLevel < 0 then RXState.CORSQLevel := 0;
    end else begin
      Dec(RXState.IFGain);
      if RXState.IFGain < 0 then RXState.IFGain := 0;
    end;
  end;

  // Make command to set new value
  if RXState.SquelchEnabled then begin
    setcmd := Set_CORSQLevel;
    par := IntToStr(RXState.CORSQLevel);
    getcmd := Get_CORSQLevel;
  end else begin
    setcmd := Set_IFGain;
    par := IntToStr(RXState.IFGain);
    getcmd := Get_IFGain;
  end;

  // Set new value
  if SendCommand(setcmd, par) then begin
    if GetReply then begin
      if SendCommand(getcmd,'') then begin
        if GetReply then begin
          if ParseStatus then begin
            // Update knob position
            UpdateIFgainAndSquelch;
            if RXstate.SquelchEnabled then
              LBIFGSQ.Caption := IntToStr(RXState.CORSQLevel)
            else
              LBIFGSQ.Caption := IntToStr(RXState.IFGain);
            DoMessages;
          end else MsgToShow := PARSESTFAILED;
        end else MsgToShow := NOREPLY;
      end else MsgToShow := COMMANDFAILED;
    end else MsgToShow := NOREPLY;
  end else MsgToShow := COMMANDFAILED;
  In_Command := FALSE;
  Handled := TRUE;
end;

// Mouse over the volume knob
procedure TRA3701.SHVOLMouseEnter(Sender: TObject);
begin
  MsgToShow := NOREMVOL;
end;

// Wheel over the volume knob
procedure TRA3701.SHVOLMouseWheel(Sender: TObject; Shift: TShiftState;
  WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
begin
  if In_ChanUpdate then exit;
  MsgToShow := NOREMVOL;
end;

// A little joke... (mouse over the "PHONES" shape)
procedure TRA3701.SHPHMouseEnter(Sender: TObject);
begin
  if In_ChanUpdate then exit;
  if KeyboardState <> STATE_POWEROFF then MsgToShow := YOUCANTPLUGHERE;
end;

//
// Menu handling
//

// "File" menu handling

// "Load state"
procedure TRA3701.MLSTATEClick(Sender: TObject);

var StateFile: FILE of TRXState;
    tmpRXState: TRXState;

begin
  OpenDialog1.InitialDir := StateDir;
  if OpenDialog1.Execute then begin
    AssignFile(StateFile,OpenDialog1.Filename);
    MsgToShow := LOADINGSTA;
    try
      Reset(StateFile);
      Read(StateFile,tmpRXState);
      CloseFile(StateFile);
    except
      Message(CANTOPENFILE);
      exit;
    end;
    RXstate := tmpRXState;
    RestoreState;
    UpdateSpinAndCombos;
    MsgToShow := DONE;
  end;
end;

// "Save state" or "Save channel"
procedure TRA3701.MSSTATEClick(Sender: TObject);

var StateFile: file of TRXState;
    SModeRX: string = '';
    sMtmp,sFtmp: string;

begin
  case RXState.Mode_RX of
    M_ISBU: SModeRX := 'ISBU';
    M_ISBL: SModeRX := 'ISBL';
    M_USB: SModeRX := 'USB';
    M_LSB: SmodeRX := 'LSB';
    M_AM:  SModeRX := 'AM';
    M_CW:  SModeRX := 'CW';
    M_FM:  SModeRX := 'FM';
    M_FSK: SModeRX := 'FSK'
  end;

  sMtmp := sModeRX;
  sFtmp := GetRXString;
  Delete(sFtmp,6,1); // Remove dot

  if Sender = MSSTATE then SaveDialog1.InitialDir := StateDir;
  if Sender = MSCHANN then SaveDialog1.InitialDir := ChannelsDir;
  SaveDialog1.FileName := sFtmp + '-' + sMtmp + '.dat';
  if SaveDialog1.Execute then begin
    AssignFile(StateFile,SaveDialog1.Filename);
    try
      Rewrite(StateFile);
      Write(StateFile,RXState);
      CloseFile(StateFile);
    except
      MsgToShow := CANTWRITESTATE;
    end;
    if Sender = MSCHANN then LoadChannelsDir;
  end;
end;

// "Manage channels"
procedure TRA3701.MMANCHClick(Sender: TObject);
begin
    UMGR.Caption := CHANMGR;
    UMGR.TFLB.Directory := ChannelsDir;
    UMGR.TFLB.UpdateFileList;
    UMGR.Tag := 1;
    UMGR.SPMOVE2CH.Hide;
    UMGR.SPMOVE2ST.Visible := TRUE;
    UMGR.Show;
end;

// "Manage States"
procedure TRA3701.MMANSTClick(Sender: TObject);
begin
    UMGR.Caption := STATEMGR;
    UMGR.TFLB.Directory := StateDir;
    UMGR.TFLB.UpdateFileList;
    UMGR.Tag := 0;
    UMGR.SPMOVE2CH.Visible := TRUE;
    UMGR.SPMOVE2ST.Hide;
    UMGR.Show;
end;

// "Restore current state"
procedure TRA3701.MRSTATEClick(Sender: TObject);

begin
    RestoreState;
    EnableAllControls;
    UpdateDisplay;
    UpdateSpinAndCombos;
    SetActiveButtons;
    DoMessages;
end;

// "Exit"
procedure TRA3701.MExitClick(Sender: TObject);
begin
    SaveConfig;
    SaveChanFile;
    SaveLastState;
    if MessageDlg(CLOSEPGM,REALLYEXIT,mtConfirmation,
                  [mbYES, mbNO],0) = mrYES then CloseProgram;
end;

//
// "Other functions" menu handling
//

// Display receiver info
procedure TRA3701.MRXINFOClick(Sender: TObject);

var i: integer;
    stmp,sBWType: string;

begin
  FRXINFO.MEMRXInfo.Lines.Clear;
  FRXINFO.MEMRXInfo.Lines.Add(ReceiverID + RXId);
  FRXINFO.MEMRXInfo.Lines.Add(RECEIVERMODEL + RXModel);
  FRXINFO.MEMRXInfo.Lines.Add(RECEIVERSN + RXSN);
  FRXINFO.MEMRXInfo.Lines.Add(FIRMWVERS + FWVersion);
  FRXINFO.MEMRXInfo.Lines.Add(LINKPARAMS + LinkParms);
  FRXINFO.MEMRXInfo.Lines.Add('');
  FRXINFO.MEMRXInfo.Lines.Add(BWCONF);
  FRXINFO.MEMRXInfo.Lines.Add(LINESEP);
  FRXINFO.MEMRXInfo.Lines.Add(TITLES);
  FRXINFO.MEMRXInfo.Lines.Add(LINESEP);
  for i := 0 to 47 do begin

    if Bandwidths[i].BWType <= 0 then break;

    case Bandwidths[i].BWType of
        1: sBWType := 'USB';
        2: sBWType := 'LSB';
        3: sBWType := 'SYM';
    end;
    FmtStr(stmp,'%4S  %6D  %3D   %5.2Fk  %5.2Fk  %6.2Fk %8.2Fk',
           [sBWType, Bandwidths[i].BWNumber, Bandwidths[i].Position,
            Bandwidths[i].FLow / 1000.0, Bandwidths[i].FHigh / 1000.0,
            Bandwidths[i].FOffset / 1000.0, Bandwidths[i].Bandwidth / 1000.0]);
    FRXINFO.MEMRXInfo.Lines.Add(stmp);
  end;
  FRXInfo.Show;
end;

// "Set scan flag for all channels"
procedure TRA3701.MSETALLSCFClick(Sender: TObject);

var i: integer;

begin
  if MUSEPROGCH.Checked then begin
    for i := 0 to 99 do begin
      RXChannels[i].ScanFlag := TRUE;
    end;
  end else begin
    if SendCommand(Change_ScanFlgs,'1') then begin
      if not GetAck then MsgToShow := NOACK;
    end else MsgToShow := COMMANDFAILED;
  end;
  UpdateLeftPanel;
end;

// "Reset scan flag for all channels"
procedure TRA3701.MRESETALLSCFClick(Sender: TObject);

var i: integer;
begin
  if MUSEPROGCH.Checked then begin
    for i := 0 to 99 do begin
      RXChannels[i].ScanFlag := FALSE;
    end;
  end else begin
    if SendCommand(Change_ScanFlgs,'0') then begin
      if not GetAck then MsgToShow := NOACK;
    end else MsgToShow := COMMANDFAILED;
  end;
  UpdateLeftPanel;
end;

procedure TRA3701.MRXFREQSCANClick(Sender: TObject);

var p: integer;
    StartFreq, Stopfreq, StepFreq,SwRate,CORC,CORL,Dir: string;

begin
  if In_Command then exit;
  In_Command := TRUE;
  if SendCommand(Get_FrSwParms, '') then begin
    if GetReply then begin
      p := Pos('SCFR',Status);
      if p > 0 then begin
        StartFreq := Copy(Status,p+4,8);
        StopFreq  := Copy(Status,p+13,8);
        StepFreq  := Copy(Status,p+22,8);
        SwRate    := Copy(Status,p+31,8);
        CORC      := Copy(Status,p+40,1);
        ffrscan.FSPSTART.Value := StrToFloat(AdjustDecimalSeparator(StartFreq))/1000;
        ffrscan.FSPSTOP.Value := StrToFloat(AdjustDecimalSeparator(StopFreq))/1000;
        ffrscan.FSPSTEP.Value := StrToFloat(AdjustDecimalSeparator(StepFreq))/1000;
        ffrscan.FSPSWRATE.Value := StrToFloat(AdjustDecimalSeparator(SwRate))/1000;
        ffrscan.CBCOR.ItemIndex := StrToInt(CORC);
      end;
    end else MsgToShow := NOREPLY;
  end else MsgToShow := COMMANDFAILED;
  In_Command := FALSE;

  // Retrieve scan direction
   if SendCommand(Get_Scan, '') then begin
     if GetReply then begin
       p := Pos('SCAN',Status);
       if p > 0 then begin
         Dir := Copy(Status,p+6,1);
         ffrscan.CBDIR.ItemIndex := StrToInt(Dir);
       end;
     end else MsgToShow := NOREPLY;
   end else MsgToShow := COMMANDFAILED;
   In_Command := FALSE;

  // Retrieve COR/SQUELCH level
  if SendCommand(Get_CORSQLevel, '') then begin
    if GetReply then begin
      p := Pos('CORL',Status);
      if p > 0 then begin
        CORL := Copy(Status,p+4,3);
        ffrscan.SPCORL.Value := StrToInt(CORL);
      end;
    end else MsgToShow := NOREPLY;
  end else MsgToShow := COMMANDFAILED;

  KeyboardState := STATE_SCAN;
  FFRSCAN.Show;
  FFRSCAN.FSPSTART.SetFocus;
end;

// "Update channel list"
procedure TRA3701.MUPFCHANLISTClick(Sender: TObject);

var ChanNo: integer;
    schn: string = '';
    SaveState: TRXState;
    SaveTBITEStatus, SaveTSMeterStatus: boolean;

begin
  // Disable all commands, save and disable timers
  SaveState := RXState;
  In_Command := TRUE;
  In_ChanUpdate := TRUE;
  DisableAllButtons;
  MOTHER.Enabled := FALSE;
  SHKNOB.Enabled := FALSE;
  SHKNOB1.Enabled := FALSE;
  SaveTBITEStatus := TBITE.Enabled;
  SaveTSMETERStatus := TSmeter.Enabled;
  TBITE.Enabled := FALSE;
  TSmeter.Enabled := FALSE;
  // set Delta mode, cycle trough all 100 channels reading status.
  // In delta mode the receiver status is unmodified, but the status
  // returned is that of the selected channel.
  if SendCommand(Set_DeltaMode, '') then begin
    DeltaMode_ON := TRUE;
    if GetReply then begin
      MsgToShow := READINGCHANLIST;
      MsgToShow := '';             // Empty line for channels counter
      for ChanNo := 0 to 99 do begin
        FmtStr(schn,'%02D',[ChanNo]);
        In_Command := TRUE;
        if SendCommand(Set_Chan,schn) then begin
          if GetReply then begin
            GetStatus;
            RXChannels[ChanNo] := RXState;
            MSG.Lines[MSG.Lines.Count-1] := (READCHANNO+schn);  //<-- This
            DoMessages;
          end else MsgToShow := NOREPLY;
        end else MsgToShow := COMMANDFAILED;
      end;
    end;
  end;
  SaveChanFile;
  MsgToShow := DONE;

  // Exit delta mode and restore program status
  if SendCommand(Recall, '') then begin
    if GetReply then begin
    end else MsgToShow := NOREPLY;
  end else MsgToShow := COMMANDFAILED;
  RXState := SaveState;
  DeltaMode_ON := FALSE;
  In_Command := FALSE;
  SetActiveButtons;
  MOTHER.Enabled := TRUE;
  SHKNOB.Enabled := TRUE;
  SHKNOB1.Enabled := TRUE;
  TBITE.Enabled := SaveTBITEStatus;
  TSmeter.Enabled := SaveTSMeterStatus;
  In_ChanUpdate := FALSE;
end;

// "Unit Confidence BITE"
procedure TRA3701.MBITEUCONFClick(Sender: TObject);

begin
  KeyBoardState := STATE_BITECONF;
  FaultFindingBITE := FALSE;
  DoBITE(2);
end;

// "Factory BITE"
procedure TRA3701.MFACTBITEClick(Sender: TObject);

begin
  KeyBoardState := STATE_BITEFACT;
  FaultFindingBITE := FALSE;
  MsgToShow := PRESSRCLTOSTOP;
  DoBITE(5);
end;

// "Fault finding BITE" (continuous, selected test)
procedure TRA3701.MFFBITEClick(Sender: TObject);

var sTestNo,par: string;
  i: integer;
  sTestNoM1: string;
  SaveTBITE, SaveTSMETER: boolean;
  sBlock, sDescr: string;
  TestExists: boolean = TRUE;

begin
  FaultFindingBITE := TRUE;
  repeat
    FEntTestNo.Show;
    repeat
      DoMessages;
    until OKPressed;
    OKPressed := FALSE;
    FEntTestNo.Hide;

    FmtStr(sTestNo, '%3.3d',[StrToInt(FEntTestNo.EDTestNo.Text)]);
    par := '4,'+sTestNo;
    DisableAllButtons;
    BRCL.Enabled := TRUE;
    In_Command := TRUE;
    TestExists := TRUE;
    if SendCommand(Set_BITE,par) then begin
      KeyboardState := STATE_BITEFF;
      if GetReply then begin
        if Pos('ERR', Status) > 0 then begin
          TestExists := FALSE;
          FEntTestNo.EDTestNo.Font.Color := clRED;
          MsgToShow := Status;
        end;
      end else MsgToShow := NOREPLY;
    end else MsgToShow := COMMANDFAILED;
  until TestExists;
  GetTestDescription(sTestNo, sBlock, sDescr);
  MsgToShow := Format(EXECUTINGTESTNO, [sTestno, sBlock, sDescr]);
  MsgToShow := PRESSRCLTOSTOP;
  repeat
    {$IFDEF NOTDEF}
      // I've not yet understood how to get test result from remote
      // and whether that is even possible in the P87915/12 firmware,
      // so this code is disabled.
      sTestNoM1 := IntToStr(StrToInt(FEntTestNo.EDTestNo.Text) - 1);
      if SendCommand(Set_BITE,'7,'+STestNom1+';QBITE') then begin
        if GetReply then begin
          // to be written
          msgtoshow := Status;
        end else MsgToShow := NOREPLY;
      end else MsgToShow := COMMANDFAILED;
      for i := 1 to 10 do begin
        Sleep(10);
        DoMessages;
      end;
      {$ELSE}
      Sleep(10);
      DoMessages;
      {$ENDIF}
    until RCLPressed;
  In_Command := FALSE;
end;

// Get detailed error list !!!NOT TESTED!!!
procedure TRA3701.MDETERRLISTClick(Sender: TObject);

var ErrNum, tmpErrNum, p1, p2, p3: integer;
    s, sErrnum, sDescr: string;
    endlist: boolean = FALSE;

begin
  errnum := 1;
  MsgToShow := STARTERRLIST;
  repeat
    s := Format('%3.3d',[errnum]);
    if SendCommand(Set_BITE,'7,'+s) then begin
      if GetReply then begin
        // Reply should be something like BITE7,nnnn,"Description";
        // or BITE7; if no error present
        // !NOT VERIFIED!
        if pos('BITE7', Status) > 0 then begin
          if Pos(',', Status) > 0 then
            sErrnum := Copy(Status,p1+1,4)
          else
            sErrNum := '';
        end;
        if TryStrToInt(sErrNum, tmpErrNum) then ErrNum := tmpErrNum;
      end else MsgToShow := NOREPLY;
    end else MsgToShow := COMMANDFAILED;
    if SendCommand(Get_BITE,'') then begin
      if GetReply then begin
        // Reply should be something like BITE7,nnnn,"Description";
        // or BITE7,0999; if no more errors
        EndList := (Pos('999', Status) > 0);
        if not EndList then begin
          // !!!CODE NOT TESTED!!!
          p1 := Pos(',', Status);
          p2 := Pos('"', Status);
          p3 := Pos(';', Status);
          sErrNum := Copy(Status, 7, 4);
          SDescr := Copy(Status, p2+1, p3-p2-2);
          MsgToShow := Format(ERRORLISTMSG, [sErrNum, sDescr]);
        end else MsgToShow := NOMOREFAULTS;
      end else MsgToShow := NOREPLY;
    end else MsgToShow := COMMANDFAILED;
    Inc(errnum);
  until EndList;
  MsgToShow := ENDERRLIST;
  if SendCommand(Recall,'') then begin
    if not GetReply then MsgToSHow := NOREPLY;
  end else MsgToShow := COMMANDFAILED;
end;

// "Band scan window" menu item
procedure TRA3701.MBSCANWClick(Sender: TObject);
begin
  FSCAN.Show;
end;

//
// "Options" menu handling
//

// Communication parameters...
procedure TRA3701.MSERPARMSClick(Sender: TObject);
begin
 fserparms.Show;
end;

// TCP server address:port
procedure TRA3701.MTCPClick(Sender: TObject);
begin
  FTCP.EDADDPORT.Text := RA3701Server.Host+':'+IntToStr(RA3701Server.Port);
  FTCP.Show;
end;

// Set kHz per turn of the simulated tuning knob
procedure TRA3701.MKHZTClick(Sender: TObject);
begin
  HzPerTurn := (Sender as TmenuItem).Tag;
  if not MTIMED.Checked then begin
    LBKHZT.Caption := IntToStr(HzPerTurn div 1000)+KHZT;
    LBKHZT.Show;
  end else LBKHZT.Hide;
end;

// "Magnify all fonts" menu item
procedure TRA3701.MMAGALLFClick(Sender: TObject);
begin
  FontMagnify;
end;

// "Enable S-Meter"
procedure TRA3701.MSmeterClick(Sender: TObject);
begin
  if MSmeter.Checked then begin
    TSmeter.Enabled := TRUE;
    TSMDISP.Enabled := TRUE;
    Enable_Status := TRUE;
  end else begin
    TSmeter.Enabled := FALSE;
    TSMDISP.Enabled := FALSE;
    Enable_Status := FALSE;
    RXstate.RFLevel := 255;
  end;
  SaveConfig;
end;

//
// "LED color"
//
// "Blue"
procedure TRA3701.MBlueClick(Sender: TObject);
begin
    MBlack.Checked := FALSE;
    MBlue.Checked := TRUE;
    MYellow.Checked := FALSE;
    LEDCOlorOFF := clGreen;
    LEDColorON := clBlue;
    sLEDColor := 'blue';
    SetColors(LEDColorON);
    SaveConfig;
end;

// "Green"
procedure TRA3701.MYellowClick(Sender: TObject);
begin
    MBlack.Checked := FALSE;
    MBlue.Checked := FALSE;
    MYellow.Checked := TRUE;
    LEDCOlorOFF := clGray;
    LEDColorON := $0000FDFF;
    sLEDColor := 'yellow';
    SetColors(LEDColorON);
    SaveConfig;
end;

// "Black"
procedure TRA3701.MBlackClick(Sender: TObject);
begin
    MBlack.Checked := TRUE;
    MBlue.Checked := FALSE;
    MYellow.Checked := FALSE;
    LEDCOlorOFF := clGray;
    LEDColorON := clBlack;
    sLEDColor := 'black';
    SetColors(LEDColorON);
    SaveConfig;
end;

// "Enable all controls"
procedure TRA3701.MENALLClick(Sender: TObject);
begin
    SaveConfig;
    if SerPort <> 0 then begin
        if MENALL.Checked then
            EnableAllControls
        else begin
            SetActiveButtons;
            //UpdateControls;
        end;
    end;
end;

// Select font magnification factor (contained in the Tag property)
procedure TRA3701.MFONTMAGNClick(Sender: TObject);
begin
    FontMagn := (Sender as TMenuItem).Tag;
    SaveConfig;
    FontMagnify;
end;

// "Show RX commands"
procedure TRA3701.MSHOWCClick(Sender: TObject);
begin
  SaveConfig;
end;

// Enable/disable continuous BITE result display
procedure TRA3701.MCONTBITEClick(Sender: TObject);
begin
  if MCONTBITE.Checked then
    TBITE.Enabled := TRUE
  else begin
    TBITE.Enabled := FALSE;
    LBBITE.Caption := '';
  end;
end;

// "Lock RX in remote" submenu entry
procedure TRA3701.MLOCKRXClick(Sender: TObject);

var Result: boolean;

begin
  if KeyboardState <> STATE_POWEROFF then begin
    if MLOCKRX.Checked then
      Result := SendCommand(Set_Remote, REMOTE_KBLOCK)
    else
      Result := SendCommand(Set_Remote,REMOTE_NOKBLOCK);

    if Result then begin
      if not (GetAck and (Status='')) then
        MsgToShow := NOACK+Status
    end;
  end;
end;

// RF Amplifier
procedure TRA3701.MRFAMPXClick(Sender: TObject);

var par: string;

begin
  par := IntToStr((Sender as TMenuItem).Tag);
  RXState.RFAmp := StrToInt(par);
  if KeyboardState <> STATE_POWEROFF then begin
    In_Command := TRUE;
    if SendCommand(Set_RFAMP, par) then begin
      if GetReply then begin
        if not ParseStatus then MsgToShow := PARSESTFAILED;
      end else MsgToShow := NOREPLY;
    end else MsgToShow := COMMANDFAILED;
    In_Command := FALSE;
    UpdateRightPanel;
  end;
end;


// Save AUX file with the current RX state
procedure TRA3701.MSAVEAUXClick(Sender: TObject);

var AUXFile: file of TRXState;

begin
  AssignFile(AUXFile,Homedir+AUXFileName);
  try
    Rewrite(AUXFile);
    Write(AUXFile,RXState);
    CloseFile(AUXFile);
  except
    MsgToShow := CANTWRITEAUXFILE;
  end;
end;

// hamlib NET rigctl-more-or-less-compatible server networking code (LNET)
// Remote commands handling code is in UParse,pas, UCapabilities.pas and UState.pas
// The server code is so far incomplete, it's a little more than the bare minimum
// needed to work with WSJTX, FLDIGI, GRIG, XDX and XLOG.
// See also the comments in UParse.pas

// Accept a connection
procedure TRA3701.RA3701ServerAccept(aSocket: TLSocket);

begin
  NumTCPConnections := NumTCPConnections+1;
  LBNETW.Visible := TRUE;
end;

  // Handle disconnect received from client
procedure TRA3701.RA3701ServerDisconnect(aSocket: TLSocket);
begin
  if NumTCPConnections > 0 then NumTCPConnections := NumTCPConnections - 1;
  if NumTCPConnections = 0 then LBNETW.Hide;
end;

// Error handling
procedure TRA3701.RA3701ServerError(const amsg: string; aSocket: TLSocket);
begin
  MsgToShow := Format(ERRORNET,[amsg]);
  UpdateDisplay;
end;

// Receive data from client
procedure TRA3701.RA3701ServerReceive(aSocket: TLSocket);

var len: integer;

begin
  len := aSocket.GetMessage(RXLine);
  {$ifdef DEBUG_NET}
  MsgToShow := 'Received '+IntToStr(len)+' bytes';
  {$endif}
  if len > 0 then begin
    {$ifdef DEBUG_NET}
    MsgToShow := 'Net RX:'+ShowControlChars(RXLine);
    {$endif}
    ParseNetMsg(RXLine,aSocket);
  end;
end;

//
// Timers
//

// Timer to get continuous BITE results
procedure TRA3701.TBITETimer(Sender: TObject);

var stmp,sres: string;
    value: integer;
    sTest, sBlock, sDescr: string;

begin
  if KeyboardState = STATE_POWEROFF then Exit;
  if In_Command then exit;
  In_Command := TRUE;
  if SendCommand(Get_BITE,'') then begin
    if GetReply then begin
      if Status <> '' then begin
        {$IFDEF TEST_USER_INTERFACE}
        Status := 'BITE0,0013,"Test user interface";';
        {$ENDIF}
        stmp := Copy(Status,5,1);
        if TryStrToInt(stmp, value) then RXState.BITELevel := value;
        stmp := Copy(Status,7,4);
        if TryStrToInt(stmp, value) then RXState.BITETest := value;
        sTest := Format('%3.3d',[value]);
      end;
    end else MsgToShow := NOREPLY;
  end else MsgToShow := COMMANDFAILED;

  if SendCommand(Get_Fault,'') then begin
    if GetReply then begin
      if Status <> '' then begin
        {$IFDEF TEST_USER_INTERFACE}
        Status := 'FAULT0;';
        {$ENDIF}
        stmp := Copy(Status,6,1);
        if stmp = ',' then stmp := Copy(Status,7);
        if TryStrToInt(stmp, value) then RXState.BITEFaults := value;
      end;
    end else MsgToShow := NOREPLY;
  end else MsgToShow := COMMANDFAILED;
  In_Command := FALSE;
  case RXState.BITEFaults of
    0: begin
      sres := PASSED;
      LBBITE.Font.Color := clWhite;
      LBBITE.Font.Style := [];
      LBFAULT.Hide;
    end;
    5: begin
      sres := SERFAULTS;
      LBBITE.Font.Color := clRed;
      LBBITE.Font.Style := [fsBold];
      LBFAULT.Show;
    end;
    otherwise sres := IntToStr(RXState.BITEFaults)+'?';
  end;
  GetTestDescription(sTest,sBlock,sDescr);
  FmtStr(stmp,EXECBITE, [RXState.BITELevel,sTest,sBlock,sDescr,sres]);
  LBBITE.Caption := stmp;
end;


//
// Menu handling
//


//
// "Channels" menu is dinamically built
// See procedure LoadChannelsDir
//

//
// "Help" menu handling
//

// "About"
procedure TRA3701.MABOUTClick(Sender: TObject);

begin
    FAbout.Show;
end;

// "Manual"
procedure TRA3701.MMANClick(Sender: TObject);

begin
    FMAN.Show;
end;

// "GNU License"
procedure TRA3701.MGNULICClick(Sender: TObject);
begin
    FGNU.Show;
end;

//
// ComboBoxes & SpinEdits
//

//  RX# spinedit
procedure TRA3701.SPRXNUMChange(Sender: TObject);

begin
  // if only 1 RX configured, ignore
  if CommParms.NumRX = 1 then begin
    SPRXNUM.Value := 1;
    exit;
  end;

  // Close comms with the previous RX
  if KeyboardState <> STATE_POWEROFF then begin
    KeyboardState := STATE_NORMAL;
    In_Command := FALSE;
    SHSWMouseUp(Sender,mbLeft,[],0,0);

    // Set new address
    CurRXNum := SPRXNUM.Value;
    CurRXAddress :=  CommParms.RXAddresses[CurRXNum];
//    MsgToShow := 'RX#: ' + IntToStr(CurRXNum) + ' Address: '+IntToStr(CurRXAddress);
    SaveConfig;

    // Open comms with the new RX
    SHSWMouseUp(Sender,mbLeft,[],0,0);
  end;
end;

// Frequency spinedit
procedure TRA3701.SPFREQChange(Sender: TObject);
begin
    {$IFDEF LINUX}
    if Update_Only or In_Command or (not RXState.Tune) then exit;
    In_Command := TRUE;
    {$ENDIF}
    {$IFDEF WINDOWS}
    // Under Windows the OnChange event fires also when it shouldnt
    if Update_Only or (RXState.Freq_RX = SPFREQ.Value) then exit;
    {$ENDIF}
    RXState.Freq_RX := SPFREQ.Value;
    SetRXFreq;
    UpdateDisplay;
    In_Command := FALSE;
end;

// MODE ComboBox
procedure TRA3701.CBXMODEChange(Sender: TObject);

begin
  if In_Command then exit;
  In_Command := TRUE;
  if SendCommand(Set_Mode, IntToStr(CBXMODE.ItemIndex)) then begin
    RXstate.Mode_RX := CBXMODE.ItemIndex;
    GetStatus;
    UpdateDisplay;
    EnableCmdKeys;
    EnableCombosAndEdits;
    SetActiveButtons;
  end else MsgToShow := COMMANDFAILED;
  In_Command := FALSE;
end;

// BW ComboBox
procedure TRA3701.CBXBWChange(Sender: TObject);

var stmp: string;
  maxBW: integer;
  BW: array[0..15] of TBandwidth;

begin
  if In_Command then exit;
  In_Command := TRUE;

  case RXstate.Mode_RX of
    M_USB: begin
      maxBW := NumUSBBW-1;
      BW := USBBandwidths;
    end;
    M_LSB: begin
      maxBW := NumLSBBW-1;
      BW := LSBBandwidths;
    end;
    otherwise begin
      maxBW := NumSYMBW-1;
      BW := SYMBandwidths;
    end;
  end;
  BWNum := CBXBW.ItemIndex;
  if BWNum > maxBW then BWNum := maxBW;
  FmtStr(stmp,'%1D,%0.2D',[BW[BWNum].BWType, BW[BWNum].BWNumber]);
  if SendCommand(Set_IFBW,stmp) then begin
    if SendCommand(Get_IFBW,'') then begin
      if GetReply then begin
        RXState.Bandwidth := BW[BWNum].FHigh-BW[BWNum].FLow;
        In_Command := FALSE;
        GetStatus;
        if not ParseStatus then MsgToShow := PARSESTFAILED;
        UpdateDisplay;
        UpdateSpinAndCombos;
        SetBWCombo;
      end
    end else MsgToShow := COMMANDFAILED;
  end else MsgToShow := COMMANDFAILED;

  In_Command := FALSE;
end;

// AGC ComboBox
procedure TRA3701.CBXAGCChange(Sender: TObject);

var par: string;

begin
  In_Command := TRUE;
  case CBXAGC.ItemIndex+1 of
    1: begin
      par := '0,0'; // SHORT
      RXState.AGCMode := 0;
      RXState.AGCTimeCon := 0;
    end;
    2: begin
      par := '0,1'; // MEDIUM
      RXState.AGCMode := 0;
      RXState.AGCTimeCon := 1;
    end;
    3: begin
      par := '0,2'; // LONG
      RXState.AGCMode := 0;
      RXState.AGCTimeCon := 2;
    end;
    4: begin
      par := '0,3'; // L2DATA
      RXState.AGCMode := 0;
      RXState.AGCTimeCon := 3;
    end;
    5: begin
      par := '0,4'; // L2NORM
      RXState.AGCMode := 0;
      RXState.AGCTimeCon := 4;
    end;
    6: begin
      par := '1,0'; // MAN ONLY
      RXState.AGCMode := 1;
      RXState.AGCTimeCon := 0;
    end;
    7: begin
      par := '2,0'; // MAN SHORT
      RXState.AGCMode := 2;
      RXState.AGCTimeCon := 0;
    end;
    8: begin
      par := '2,1'; // MAN MEDIUM
      RXState.AGCMode := 2;
      RXState.AGCTimeCon := 1;
    end;
    9: begin
      par := '2,2'; // MAN LONG
      RXState.AGCMode := 2;
      RXState.AGCTimeCon := 2;
    end;
   10: begin
     par := '2,3'; // MAN L2DATA
     RXState.AGCMode := 2;
     RXState.AGCTimeCon := 3;
   end;
   11: begin
     par := '2,4'; // MAN L2NORM
     RXState.AGCMode := 2;
     RXState.AGCTimeCon := 4;
   end;
  end;

  if SendCommand(Set_AGC, par) then begin
    if GetReply then begin
      RXstate.AGCMode := CBXAGC.ItemIndex+1;
    end;
  end;
  EnableCmdKeys;
  EnableCombosAndEdits;
  UpdateDisplay;
  UpdateSpinAndCombos;
  In_Command := FALSE;
end;

// Mouse wheel on frequency spinedit
procedure TRA3701.SPFREQMouseWheel(Sender: TObject; Shift: TShiftState;
    WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);

begin
    If WheelDelta > 0 then SPFREQ.Value := SPFREQ.Value + 1;
    If WheelDelta < 0 then SPFREQ.Value := SPFREQ.Value - 1;
    Handled := TRUE;
end;

// Alternative frequency setting way: left/right click on the frequency figures
// Clicking on the decimal point rewrites displayed frequency
procedure TRA3701.RXfreqMouseUp(Sender: TObject; Button: TMouseButton;
        Shift: TShiftState; X, Y: Integer);

var step: integer;

begin
    if In_Command or In_RXFreqMouseUp or (not RXstate.Tune) then exit;

    {$IFDEF TEST_FREQMOUSEUP}
    MsgToShow := 'X='+IntToStr(X)+' Y='+IntTostr(Y);
    {$ENDIF}

    In_RXFreqMouseUp := TRUE;
    step := (Sender as TLabel).tag;
    if step < 0 then step := 0;

    if Button = mbLeft then begin
        if RXState.Freq_rx - step >= MINRXF then
	   RXState.Freq_rx := RXState.Freq_rx-step;
	end;
    if Button = mbRight then begin
        if RXState.Freq_rx + step <= MAXRXF then
	   RXState.Freq_rx := RXState.Freq_rx+step;
	end;
    if not SetRXFreq then
        MsgToShow := COMMANDFAILED;
    DispRXf;

    In_RXFreqMouseUp := FALSE;
end;

// Wheel over frequency display
procedure TRA3701.RXFreqMouseWheel(Sender: TObject; Shift: TShiftState;
    WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);

    var step: integer = -1;

begin
    if In_Wheel or In_Command or (not RXState.Tune) then exit;
    In_Wheel := TRUE;
    {$IFDEF TEST_FREQMOUSEWHEEL}
    MsgToShow := 'X='+IntToStr(MousePos.X)+' Y='+IntTostr(MousePos.Y);
    {$ENDIF}

    step := (Sender as TLabel).Tag;
    if step < 0 then step := 0;

    // Mouse wheel over RX frequency display
    if WheelDelta > 0 then begin
        if step >= 0 then begin
	        if RXState.Freq_rx + step <= MAXRXF then begin
                RXState.Freq_rx := RXState.Freq_rx + step;
                if not SetRXFreq then MsgToShow := COMMANDFAILED;
            end else MsgToShow := RXCANTTUNE;
        end;
    end;
    if WheelDelta < 0 then begin
        if step >= 0 then begin
	        if RXState.Freq_rx - step >= MINRXF then begin
                RXState.Freq_rx := RXState.Freq_rx-step;
                if not SetRXFreq then MsgToShow := COMMANDFAILED;
            end else MsgToShow := RXCANTTUNE;
        end;
    end;
    DispRXF;
//    UpdateDisplay;
    SpinEditUpdateOnly(SPFREQ, RXstate.Freq_RX);

    Handled := TRUE;
    In_Wheel := FALSE;
end;

// Mouse selection of channel number

// Mouse click on channel display: left=DOWN, right=UP
procedure TRA3701.CHANMouseUp(Sender: TObject; Button: TMouseButton;
    Shift: TShiftState; X, Y: Integer);

var s: string;
    delta: integer;

begin
    if In_Command or In_ChanMouseUp then exit;
    In_ChanMouseUp := TRUE;
    delta := (Sender as TLabel).Tag;
    if Button = MBLEFT then CurChan := CurCHan-delta;
    if Button = MBRIGHT then CurChan := CurChan+delta;
    if CurChan > 99 then CurChan := CurChan-100;
    if CurChan < 0  then CurChan := CurChan+100;

    s := Format('%2.2D',[CurChan]);
    PutChanString(s);
    LastChan := CurChan;
    if KeyboardState = STATE_SELCHAN then
        RXState := RXChannels[CurChan];
    UpdateDisplay;
    In_ChanMouseUp := FALSE;
end;

// Mouse Wheel on channel display
procedure TRA3701.CHANMouseWheel(Sender: TObject; Shift: TShiftState;
    WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);

var s: string;
    delta: integer;

begin
    if In_Command or In_ChanMouseWheel then exit;

    {$IFDEF TEST_CHANMOUSEWHEEL}
    MsgToShow := 'X='+IntToStr(MousePos.X)+' Y='+IntTostr(MousePos.Y);
    {$ENDIF}

    In_ChanMouseWheel := TRUE;

    delta := (Sender as TLabel).Tag;

    if (WheelDelta>0) then CurChan := (CurChan+delta);
    if (WheelDelta<0) then CurChan := (CurChan-delta);
    if CurChan > 99 then CurChan := CurChan-100;
    if CurChan < 0 then  CurChan := CurChan+100;
    s := Format('%2.2D',[CurChan]);
    PutChanString(s);
    LastChan := CurChan;
    if KeyboardState = STATE_SELCHAN then
        RXState := RXChannels[CurChan];
    UpdateDisplay;
    In_ChanMouseWheel := FALSE;
    Handled := TRUE;
end;

// Allows to use mouse wheel for tuning
procedure TRA3701.FormMouseWheel(Sender: TObject; Shift: TShiftState;
        WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);

var s: string;

begin
  if In_Wheel or In_Command then exit;
  In_Wheel := TRUE;
  {$IFDEF TEST_FORMMOUSEWHEEL}
  MsgToShow := 'X='+IntToStr(MousePos.X)+' Y='+IntTostr(MousePos.Y);
  {$ENDIF}
  case KeyboardState of
    STATE_NORMAL: begin
      if RXstate.Tune then begin
        if WheelDelta > 0 then begin
          if RXState.Freq_rx+RXState.Freq_step <= MAXRXF then begin
            RXState.Freq_rx := RXState.Freq_rx + RXState.Freq_step;
            if SetRXFreq then begin
              DispRXF;
            end else MsgToShow := COMMANDFAILED;
          end else MsgToShow := RXCANTTUNE;
        end;
        if WheelDelta < 0 then begin
          if RXState.Freq_rx-RXState.Freq_step >= MINRXF then begin
            RXState.Freq_rx := RXState.Freq_rx - RXState.Freq_step;
            if SetRXFreq then begin
              DispRXf;
            end else MsgToShow := COMMANDFAILED;
          end else MsgToShow := RXCANTTUNE;
        end;
      end;
      UpdateLeftPanel;
    end;
    STATE_SELCHAN: begin
      if (WheelDelta>0) then CurChan := (CurChan+1);
      if (WheelDelta<0) then CurChan := (CurChan-1);
      if CurChan < 0 then CurChan := 99;
      if CurChan > 99 then CurChan := 0;
      s := Format('%2.2D',[CurChan]);
      PutChanString(s);
      if MUSEPROGCH.Checked then begin
        RXState := RXChannels[CurChan];
        UpdateDisplay;
        LastChan := CurChan;
      end else begin
        if In_ChanMouseWheel then exit;
        In_ChanMouseWheel := TRUE;
        // We are still in delta mode, retrieve channel info and display them
        // The receiver continues to receive at the previous frequency
        if SendCommand(Set_Chan,s) then begin
          if GetReply then begin
            if SendCommand(Get_All, '') then begin
                if GetReply then begin
                  if not ParseStatus then MsgToShow := PARSESTFAILED;
                end else MsgToShow := NOREPLY;
            end else MsgToShow := COMMANDFAILED;
          end else MsgToShow := NOREPLY;
        end else MsgToShow := COMMANDFAILED;
      end;
      UpdateDisplay;
      In_ChanMouseWheel := FALSE;
    end;
    STATE_STOCHAN: begin
      if (WheelDelta>0) then CurChan := (CurChan+1);
      if (WheelDelta<0) then CurChan := (CurChan-1);
      if CurChan < 0 then CurChan := 99;
      if CurChan > 99 then CurChan := 0;
      s := Format('%2.2D',[CurChan]);
      PutChanString(s);
      if MUSEPROGCH.Checked then begin
        RXState := RXChannels[CurChan];
        UpdateDisplay;
        LastChan := CurChan;
      end else begin
        if In_ChanMouseWheel then exit;
        In_ChanMouseWheel := TRUE;
        // We are still in delta mode, retrieve channel info and display them
        // The receiver continues to receive at the previous frequency
        if SendCommand(Set_Chan,s) then begin
          if GetReply then begin
            if SendCommand(Get_All, '') then begin
                if GetReply then begin
                  if not ParseStatus then MsgToShow := PARSESTFAILED;
                end else MsgToShow := NOREPLY
            end else MsgToShow := COMMANDFAILED;
          end else MsgToShow := NOREPLY;
        end else MsgToShow := COMMANDFAILED;
      end;
      UpdateDisplay;
      In_ChanMouseWheel := FALSE;
    end;
  end;
  Handled := TRUE;
  In_Wheel := FALSE;
end;

//
// Timers
//

// Status timer. Get signal level every 500 ms.
procedure TRA3701.TSmeterTimer(Sender: TObject);

begin
  if Enable_Status and (KeyboardState <> STATE_POWEROFF) then GetLevel;
end;

// Display signal level every 300 ms
procedure TRA3701.TSMDISPTimer(Sender: TObject);

var level,i: integer;

begin
  {$IFDEF TEST_USER_INTERFACE}
  RXState.RFLevel := 100+random(50);
  RXState.AFLevel := 100+random(75);
  {$ENDIF}
  if RXState.MeterMode then
    case RXstate.RFLevel of
        000..097: level := 12;
        098..109: level := 11;
        110..120: level := 10;
        121..132: level := 9;
        133..144: level := 8;
        145..155: level := 7;
        156..167: level := 6;
        168..179: level := 5;
        180..190: level := 4;
        191..202: level := 3;
        203..214: level := 2;
        215..225: level := 1;
        otherwise level := 0;
    end
  else
    case RXstate.AFLevel of
        000..067: level := 0;
        068..075: level := 1;
        076..084: level := 2;
        085..095: level := 3;
        096..107: level := 4;
        108..120: level := 5;
        121..134: level := 6;
        135..151: level := 7;
        152..170: level := 8;
        171..190: level := 9;
        191..214: level := 10;
        215..240: level := 11;
        otherwise level := 12;
    end;
  for i := 1 to 12 do begin
    if i <= level then
      (Smeter[i] as TShape).Show
    else
      (Smeter[i] as TShape).Hide;
  end;
end;

// Timer to update main window if required
procedure TRA3701.TUPDDISPTimer(Sender: TObject);
begin
  // Update displayed data
  if PanelUpdateRequired then begin
    PanelUpdateRequired := FALSE;
    UpdateDisplay;
  end;
end;


// Things to do at startup
initialization

begin
  // Use internal Lazarus versioning
  if GetProgramVersion(ProgVersion) then begin
    Major := ProgVersion[1];
    Minor := ProgVersion[2];
    Revision := ProgVersion[3];
    Build := ProgVersion[4];
    // Make version strings
    ShortVersion := 'v'+IntTostr(Major)+'.'+IntTostr(Minor)+IntToStr(revision)+
                 ' build '+IntToStr(Build);
    LongVersion := ShortVersion+LineEnding+COPYRIGHTST;
  end else begin
    ShowMessage(GETPROGVFAIL);
    ShortVersion := NOVERSINFO;
    LongVersion := ShortVersion+LineEnding+COPYRIGHTST;
  end;

  {$IFDEF LINUX}
  HomeDir := GetUserDir+'.RA3701Control/';
  StateDir := HomeDir+'States/';
  ChannelsDir := HomeDir+'Channels/';
  BandScansdir := Homedir+'BandScans/';
  DocDir := '/usr/share/doc/ra3701control/';
  GPLv3Dir := '/usr/share/doc/ra3701control/GPLv3/';
  {$ENDIF}
  {$IFDEF WINDOWS}
  // Use the executable path to discover the location of the
  // RA3701Control directory.
  HomeDir := ExtractFilePath(Application.ExeName);
  StateDir := HomeDir+'States\';
  ChannelsDir := HomeDir+'Channels\';
  BandScansdir := Homedir+'BandScans\';
  DocDir := HomeDir+'doc\';
  GPLv3Dir := DocDir+'GPLv3\';
  {$ENDIF}
  if not DirectoryExists(HomeDir) then CreateDir(HomeDir);
  if not DirectoryExists(StateDir) then CreateDir(StateDir);
  if not DirectoryExists(BandsDir) then CreateDir(BandsDir);
  if not DirectoryExists(BandScansDir) then CreateDir(BandScansDir);
  if not DirectoryExists(ChannelsDir) then CreateDir(ChannelsDir);

  // Check manual and license files
  if not (FileExists(DocDir+'Manual-en.txt') and
          FileExists(GPLv3Dir+'gpl-3.0.txt')) then
    ErrMsg := DOCFILESNOTF+LineEnding+CHECKINST;
  {$IFDEF TEST_BROKENINSTALLMSG}
  errmsg := 'Test broken install';
  {$ENDIF}
  // If installation is severely broken, notify user and close program now.
  // Moved to FormCreate since here Application is not running yet
  // although in previous versions of Lazarus it worked also here.
  // In FormCreate we check is errmsg is not empty and, if yes, we
  // display it and clese the program.
end;

end.

