{
    This file is part of RA3701Control

    RACAL RA3701 Control program

    Copyright (C) 2025 G. Perotti, I1EPJ, i1epj@aricasale.it

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/.

                                    * * *
    Communication parameters select dialog
}

unit userparms;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, FileUtil, Forms, Controls, Graphics, Dialogs, StdCtrls,
  Grids, Math;

type

  { TFSerParms }

  TFSerParms = class(TForm)
    BOK: TButton;
    BCANCEL: TButton;
    CBCRC: TComboBox;
    CBNADCH: TComboBox;
    CBNRX: TComboBox;
    CBPORT: TComboBox;
    CBPROTOCOL: TComboBox;
    CBLCC: TComboBox;
    CBSPEED: TComboBox;
    CBPARITY: TComboBox;
    LBCRC: TLabel;
    LBNADDRC: TLabel;
    LBNRX: TLabel;
    LBRXCONF: TLabel;
    LBPORT: TLabel;
    LBPROTOCOL: TLabel;
    LBLCC: TLabel;
    LBSPEED: TLabel;
    LBPARITY: TLabel;
    SGRXADDRESSES: TStringGrid;
    procedure BCANCELClick(Sender: TObject);
    procedure BOKClick(Sender: TObject);
    procedure CBNADCHChange(Sender: TObject);
    procedure CBNRXChange(Sender: TObject);
    procedure FormShow(Sender: TObject);
  private
    { private declarations }
  public
    { public declarations }
  end;

  // The communication parameters in a record
  TCommParms = record
    SerPortName: string;
    SerPortSpeed: integer;
    SerPortParity: integer;
    SerPortProtocol: string;
    NumLCCChars: integer;
    NumAddrChars: integer;
    CRCEnabled: boolean;
    NumRX: integer;
    RXAddresses: array[1..99] of integer;
  end;

var
  FSerParms: TFSerParms;
  MaxNumRX: integer;
  CommParms: TCommParms;

implementation

{$R *.lfm}

{ TFSerParms }

uses ura3701;

// OK Button handling.
procedure TFSerParms.BOKClick(Sender: TObject);

var i,addr,MaxRXaddr: integer;
  err,Ok: boolean;

begin
  // Reads and checks parameters and stores them in the CommParms record.
  with CommParms do begin
    {$IFDEF LINUX}
    if pos('/dev',CBPORT.Text) = 0 then
      SerPortName := '/dev/'+CBPORT.Text
    else
      SerPortName := CBPORT.Text;
    {$ENDIF}
    {$IFDEF WINDOWS}
    // The \\.\ syntax is optional for ports 1..9 but is needed
    // for ports > 9, so insert it anyway if not present
    if pos('\\.\', CBPORT.Text) = 0 then
      SerPortName := '\\.\' + CBPORT.Text
    else
      SerPortName := CBPORT.Text;
    {$ENDIF}
    SerPortSpeed := StrToInt(CBSPEED.Text);
    SerPortParity := CBPARITY.ItemIndex;
    SerPortProtocol := CBPROTOCOL.Text;
    NumLCCChars := CBLCC.ItemIndex;
    NumAddrChars := CBNADCH.ItemIndex;
    MaxRXAddr := Trunc(Power(10,NumAddrChars));
    CRCEnabled :=  (CBCRC.ItemIndex > 0);
    NumRX := CBNRX.ItemIndex + 1;
    // Check for valid RX adddresses
    OK:= TRUE;
    if NumAddrChars > 0 then begin
      for i := 1 to NumRX do begin
        err := FALSE;
        if TryStrToInt(SGRXADDRESSES.Cells[1,i],addr) then begin
          if addr < MaxRXAddr then begin
            RXaddresses[i] := addr
          end else err := TRUE;
        end else err := TRUE;
        if err then begin
          SGRXADDRESSES.Cells[1,i] := SGRXADDRESSES.Cells[1,i] + '??';
          Ok := FALSE;
        end;
      end;
      if not Ok then exit;
    end;
  end;
  RA3701.SaveConfig;
  FSerParms.Hide;
end;

// Cancel button handling.
// Hide window without doing anything
procedure TFSerParms.BCANCELClick(Sender: TObject);
begin
  FSerParms.Hide;
end;

// Handles change in number of address characters
procedure TFSerParms.CBNADCHChange(Sender: TObject);

var i: integer;

begin
  case CBNADCH.ItemIndex of
    0: begin
      SGRXADDRESSES.Enabled := FALSE;
      CBNRX.Enabled := FALSE;
      MaxNumRX := 1;
    end;
    1: begin
      SGRXADDRESSES.Enabled := TRUE;
      CBNRX.Enabled := TRUE;
      MaxNumRX := 9;
      SGRXADDRESSES.RowCount := 10;
      for i := 1 to StrToInt(CBNRX.Text) do SGRXADDRESSES.Cells[0,i] := IntToStr(i);
    end;
    2: begin
      SGRXADDRESSES.Enabled := TRUE;
      CBNRX.Enabled := TRUE;
      MaxNumRX := 99;
      SGRXADDRESSES.RowCount := 100;
      for i := 1 to StrToInt(CBNRX.Text) do SGRXADDRESSES.Cells[0,i] := IntToStr(i);
    end;
    otherwise begin
      SGRXADDRESSES.Enabled := FALSE;
      MaxNumRX := 1;
    end;
  end;
  SGRXADDRESSES.RowCount := StrToInt(CBNRX.Text)+1;
  CBNRX.Clear;
  for i := 1 to MaxNumRX do CBNRX.Items.Add(IntToStr(i));
  CBNRX.ItemIndex := 0;
end;

// Handles changes in number of receivers
procedure TFSerParms.CBNRXChange(Sender: TObject);

var i: integer;

begin
  SGRXADDRESSES.RowCount := StrToInt(CBNRX.Text)+1;
  for i := 1 to StrToInt(CBNRX.Text) do SGRXADDRESSES.Cells[0,i] := IntToStr(i);
end;

// Things to do when form is show.
// Reads CommsParms record and set controls default values.
procedure TFSerParms.FormShow(Sender: TObject);

var i: integer;

begin
  SGRXADDRESSES.Cells[0,0] := 'RX#';
  SGRXADDRESSES.Cells[1,0] := 'Addr';

  {$IFDEF WINDOWS}
  // Set serial port names for Windows
  CBPORT.Clear;
  CBPORT.AddItem('\\.\COM1',nil);
  CBPORT.AddItem('\\.\COM2',nil);
  CBPORT.AddItem('\\.\COM3',nil);
  CBPORT.AddItem('\\.\COM4',nil);
  {$ENDIF}

  // Set controls value from CommParms record
  CBPORT.Text := CommParms.SerPortName;
  CBSPEED.Text := IntToStr(CommParms.SerPortSpeed);
  CBPARITY.ItemIndex := CommParms.SerPortParity;
  CBPROTOCOL.Text := CommParms.SerPortProtocol;
  CBLCC.ItemIndex := CommParms.NumLCCChars;
  CBNADCH.ItemIndex := CommParms.NumAddrChars;
  if CommParms.CRCEnabled then
    CBCRC.ItemIndex := 1
  else
    CBCRC.ItemIndex := 0;

  CBNRX.Text := IntToStr(CommParms.NumRX);
  SGRXADDRESSES.RowCount := CommParms.NumRX + 1;

  for i := 1 to CommParms.NumRX do begin
    SGRXADDRESSES.Cells[0,i] := IntToStr(i);
    SGRXADDRESSES.Cells[1,i] := IntToStr(CommParms.RXAddresses[i]);
  end;

  if CBNADCH.ItemIndex = 0 then begin
    LBNRX.Enabled := FALSE;
    LBRXCONF.Enabled := FALSE;
    SGRXADDRESSES.Enabled := FALSE;
  end else begin
    LBNRX.Show;
    SGRXADDRESSES.Enabled := TRUE;
  end;

  {$IFNDEF ENABLE_ALL_COMMPARMS}
  // Disable configuration controls not yet handled
  // Currently only port, speed and parity are honored
  CBPROTOCOL.Enabled := FALSE;
  CBLCC.Enabled := FALSE;
  CBCRC.Enabled := FALSE;
  CBNADCH.Enabled := FALSE;
  CBNRX.Enabled := FALSE;
  LBNRX.Enabled := FALSE;
  LBNADDRC.Enabled := FALSE;
  LBCRC.Enabled := FALSE;
  LBLCC.Enabled := FALSE;
  LBPROTOCOL.Enabled := FALSE;
  LBRXCONF.Enabled := FALSE;
  SGRXADDRESSES.Enabled := FALSE;
  {$ENDIF}
end;

end.

